use adw::prelude::*;
use gtk::glib;
use std::rc::Rc;
use rust_decimal::prelude::*;
use crate::utils;
use crate::pages_af;
use crate::ie_csvs;

pub const COLUMNAS: &[&str] = &["Carné", "Apellidos", "Nombre", "Forma cotización", "Cuota", "Fecha nacimiento", "Teléfono", "Correo electrónico", "Dirección", "Localidad", "Barrio", "Información adicional", "Ocupación", "Sección", "Centro de trabajo", "Dirección centro", "Información adicional centro", "Fecha afiliación", "Mes cotizado", "Fecha baja"];

pub fn af(conn: &Rc<rusqlite::Connection>, stack: &adw::ViewStack, nav: &adw::NavigationView, window: &adw::ApplicationWindow, searchentry: &gtk::SearchEntry, store: &gtk::gio::ListStore, store_cajas: &gtk::gio::ListStore)
{
    let vbox = gtk::Box::new(gtk::Orientation::Vertical, 0);
    let hbox = gtk::Box::builder()
        .orientation(gtk::Orientation::Horizontal)
        .margin_top(10)
        .margin_bottom(10)
        .homogeneous(true)
        .build();
    let dd = gtk::DropDown::from_strings(&["Altas", "Bajas"]);
    dd.set_tooltip_text(Some("Seleccionar Altas/Bajas"));
    dd.set_halign(gtk::Align::Center);
    hbox.append(&dd);
    let mostrar = gtk::MenuButton::builder()
        .label("Mostrar")
        .tooltip_text("Seleccionar columnas visibles")
        .halign(gtk::Align::Center)
        .build();
    hbox.append(&mostrar);
    let filtrosb = gtk::MenuButton::builder()
        .label("Filtros")
        .tooltip_text("Seleccionar filtros")
        .halign(gtk::Align::Center)
        .build();
    hbox.append(&filtrosb);
    vbox.append(&hbox);
    let columnview = gtk::ColumnView::builder()
        .reorderable(false)
        .enable_rubberband(true)
        .build();
    let scrolled_window = gtk::ScrolledWindow::builder()
        .child(&columnview)
        .vexpand(true)
        .build();
    vbox.append(&scrolled_window);
    let actbar = gtk::ActionBar::new();
    let maf = gtk::MenuButton::new();
    actbar.pack_start(&maf);
    let nf = gtk::Label::new(None);
    actbar.set_center_widget(Some(&nf));
    let cont_af = adw::ButtonContent::builder()
        .icon_name("contact-new-symbolic")
        .label("Afiliar")
        .build();
    let af = gtk::Button::builder()
        .child(&cont_af)
        .css_classes(["suggested-action"])
        .build();
    actbar.pack_end(&af);
    let cont_cot = adw::ButtonContent::builder()
        .icon_name("view-grid-symbolic")
        .label("Cotizaciones")
        .build();
    let cot = gtk::Button::builder()
        .child(&cont_cot)
        .css_classes(["raised"])
        .build();
    actbar.pack_end(&cot);
    vbox.append(&actbar);
    stack.add_titled_with_icon(&vbox, Some("af"), "Afiliación", "system-users-symbolic");
    let conn_clone = conn.clone();
    let dd_clone = dd.clone();
    let filtrosb_clone = filtrosb.clone();
    let searchentry_clone = searchentry.clone();
    let custfilter = gtk::CustomFilter::new(move |item| {
        filtros(&item, &dd_clone, &filtrosb_clone, &searchentry_clone)
    });
    let fstore = gtk::FilterListModel::new(Some(store.clone()), Some(custfilter.clone()));
    let sorter = columnview.sorter();
    let slm = gtk::SortListModel::new(Some(fstore.clone()), sorter);
    let sel = gtk::MultiSelection::new(Some(slm.clone()));
    columnview.set_model(Some(&sel));
    let pop_ctx = crea_menu_contextual(&conn_clone, &nav, &window, &store, &sel, &store_cajas);
    pop_ctx.set_parent(&columnview);
    crea_columnas(&columnview, &sel, &pop_ctx);
    let nf_clone = nf.clone();
    let sel_clone = sel.clone();
    let window_clone = window.clone();
    fstore.connect_items_changed(move |fstore,_,_,_| {
        nfilas_nsel(&fstore, &sel_clone, &nf_clone, &window_clone);
    });
    crea_menu_filtros(&filtrosb, &custfilter);
    crea_menu(&maf, &conn_clone, &nav, &window, &store, &slm, &store_cajas, &columnview);
    crea_menu_mostrar(&mostrar, &columnview, &window);
    crea_lista(&store, &conn);
    if fstore.n_items() > 0 {
        columnview.scroll_to(0, None, gtk::ListScrollFlags::NONE, None);
    }
    let window_clone = window.clone();
    sel.connect_selection_changed(move |sel, _, _| {
        nfilas_nsel(&fstore, &sel, &nf, &window_clone);
    });
    
    let gc = gtk::GestureClick::new();
    gc.set_button(gtk::gdk::BUTTON_PRIMARY);
    columnview.add_controller(gc.clone());
    gc.connect_pressed(move |gc, _, _, _| {
        let modifiers = gc.current_event_state();
        if !modifiers.contains(gtk::gdk::ModifierType::CONTROL_MASK) && !modifiers.contains(gtk::gdk::ModifierType::SHIFT_MASK){
            sel.unselect_all();    
        }
    });
    let window_clone = window.clone();
    let custfilter_clone = custfilter.clone();
    let columnview_clone = columnview.clone();
    dd.connect_selected_notify(move |dd| {
        let binding = columnview_clone.columns().item(19).unwrap();
        let col_fbaja = binding.downcast_ref::<gtk::ColumnViewColumn>().unwrap();
        let binding = mostrar.popover().unwrap();
        let pop_mostrar = binding.downcast_ref::<gtk::PopoverMenu>().unwrap();
        let binding = pop_mostrar.menu_model().unwrap();
        let menu_mostrar = binding.downcast_ref::<gtk::gio::Menu>().unwrap();
        //let binding = pop_ctx.menu_model().unwrap();
        let menu_ctx = pop_ctx.menu_model().and_downcast::<gtk::gio::Menu>().unwrap();
        if dd.selected() == 0 { // Altas
            col_fbaja.set_visible(false);
            menu_mostrar.remove(19);
            dd.remove_css_class("warning");
            menu_ctx.remove(2);
            menu_ctx.remove(2);
            menu_ctx.insert(2, Some("Causar baja"), Some("win.Causarbaja_af_ctx"));
        } else { // Bajas
            menu_mostrar.append(Some("Fecha baja"), Some("win.fechabaja"));
            let st = window_clone.action_state("fechabaja").unwrap().get::<bool>().unwrap();
            col_fbaja.set_visible(st);
            dd.add_css_class("warning");
            menu_ctx.remove(2);
            menu_ctx.insert(2, Some("Reincorporar"), Some("win.Reincorporar_af_ctx"));
            menu_ctx.insert(3, Some("Eliminar"), Some("win.Eliminar_af_ctx"));
        }
        custfilter_clone.changed(gtk::FilterChange::Different);
    });
    let custfilter_clone = custfilter.clone();
    searchentry.connect_search_changed(move |_| {
        custfilter_clone.changed(gtk::FilterChange::Different);
    });
    let conn_clone = conn.clone();
    let nav_clone = nav.clone();
    let store_clone = store.clone();
    cot.connect_clicked(move |_| {
        pages_af::cotizaciones(&conn_clone, &nav_clone, &store_clone);
    });
    let conn_clone = conn.clone();
    let nav_clone = nav.clone();
    let window_clone = window.clone();
    let store_clone = store.clone();
    af.connect_clicked(move |_| {
        pages_af::afiliar(&conn_clone, &nav_clone, &window_clone, &store_clone, None, Some(&columnview));
    });
}

fn crea_columnas(columnview: &gtk::ColumnView, sel: &gtk::MultiSelection, pop: &gtk::PopoverMenu) {
    let cn = COLUMNAS;
    let settings = gtk::gio::Settings::new("org.cntait.tesoreria.afiliacion.ordenar");
    let mut j: usize = settings.int("columna").try_into().unwrap();
    if j > cn.len()-1 { j = 1; }
    let asc_desc = settings.boolean("ascendente");
    let sortdir = if asc_desc { gtk::SortType::Ascending } else { gtk::SortType::Descending };
    for i in 0..cn.len() {
        let colfactory = gtk::SignalListItemFactory::new();
        let sel_clone = sel.clone();
        let pop_clone = pop.clone();
        let columnview_clone = columnview.clone();
        colfactory.connect_setup(move |_factory, item| {
            let item = item.downcast_ref::<gtk::ListItem>().unwrap();
            let label = gtk::Label::new(None);
            label.set_halign(gtk::Align::Start);
            let cbox = gtk::Box::new(gtk::Orientation::Horizontal, 0);
            cbox.append(&label);
            item.set_child(Some(&cbox));
            
            let gc = gtk::GestureClick::new();
            gc.set_button(gtk::gdk::BUTTON_SECONDARY);
            cbox.add_controller(gc.clone());
            let item_clone = item.clone();
            let sel_clone = sel_clone.clone();
            let pop_clone = pop_clone.clone();
            let columnview_clone = columnview_clone.clone();
            gc.connect_pressed(move |_, _, x, y| {
                let pres = item_clone.item().unwrap();
                let model = sel_clone.model().unwrap();
                for i in 0..model.n_items() {
                    if pres == model.item(i).unwrap() {
                        if !sel_clone.is_selected(i) {
                            sel_clone.select_item(i, true);
                        }
                        break;
                    }
                }
                let npo = cbox.compute_point(&columnview_clone, &gtk::graphene::Point::new(x as f32, y as f32));
                pop_clone.set_pointing_to(Some(&gtk::gdk::Rectangle::new(npo.unwrap().x() as i32, npo.unwrap().y() as i32, 0, 0)));
                pop_clone.popup();
            });
        });
        colfactory.connect_bind(move |_factory, item| {
            let item = item.downcast_ref::<gtk::ListItem>().unwrap();
            let cbox = item.child().and_downcast::<gtk::Box>().unwrap();
            let label = cbox.first_child().and_downcast::<gtk::Label>().unwrap();
            let strl = item.item().and_downcast::<gtk::StringList>().unwrap();
            let strl_n = strl.string(i.try_into().unwrap()).unwrap();
            label.set_label(&strl_n);
        });
        let col = gtk::ColumnViewColumn::new(Some(&cn[i]), Some(colfactory));
        columnview.append_column(&col);
        let sorter_c;
        if i == 4 { // Cuota
            sorter_c = Some(gtk::CustomSorter::new(move |a, b| {
                let sa = a.downcast_ref::<gtk::StringList>().unwrap().string(i.try_into().unwrap()).unwrap();
                let quotea;
                if sa == "" {
                    quotea = 0.0;
                } else {
                    quotea = sa.split_whitespace().next().unwrap().replace(",", ".").parse::<f32>().unwrap();
                }
                let sb = b.downcast_ref::<gtk::StringList>().unwrap().string(i.try_into().unwrap()).unwrap();
                let quoteb;
                if sb == "" {
                    quoteb = 0.0;
                } else {
                    quoteb = sb.split_whitespace().next().unwrap().replace(",", ".").parse::<f32>().unwrap();
                }
                if quotea > quoteb { gtk::Ordering::Larger }
                else if quotea < quoteb { gtk::Ordering::Smaller }
                else { gtk::Ordering::Equal }
            }));
        } else if i == 5 || i == 17 || i == 19 { // Fechas
            sorter_c = Some(gtk::CustomSorter::new(move |a, b| {
                let sa = a.downcast_ref::<gtk::StringList>().unwrap().string(i.try_into().unwrap()).unwrap();
                let datea;
                if sa == "" {
                    datea = sa.to_string();
                } else {
                    let daya = &sa[0..2];
                    let montha = &sa[3..5];
                    let yeara = &sa[6..10];
                    datea = format!("{}{}{}", yeara, montha, daya);
                }
                let sb = b.downcast_ref::<gtk::StringList>().unwrap().string(i.try_into().unwrap()).unwrap();
                let dateb;
                if sb == "" {
                    dateb = sb.to_string();
                } else {
                    let dayb = &sb[0..2];
                    let monthb = &sb[3..5];
                    let yearb = &sb[6..10];
                    dateb = format!("{}{}{}", yearb, monthb, dayb);
                }
                datea.cmp(&dateb).into()
            }));
            
        } else if i == 18 { // Mes cotizado
            sorter_c = Some(gtk::CustomSorter::new(move |a, b| {
                let sa = a.downcast_ref::<gtk::StringList>().unwrap().string(i.try_into().unwrap()).unwrap();
                let cota;
                if sa == "" {
                    cota = sa.to_string();
                } else {
                    let mut spla = sa.split_whitespace();
                    let ma = spla.next().unwrap();
                    let indexa = utils::MESES.iter().position(|&r| r == ma).unwrap();
                    let ya = spla.next().unwrap();
                    cota = format!("{}{:02}", ya, indexa);
                }
                let sb = b.downcast_ref::<gtk::StringList>().unwrap().string(i.try_into().unwrap()).unwrap();
                let cotb;
                if sb == "" {
                    cotb = sb.to_string();
                } else {
                    let mut splb = sb.split_whitespace();
                    let mb = splb.next().unwrap();
                    let indexb = utils::MESES.iter().position(|&r| r == mb).unwrap();
                    let yb = splb.next().unwrap();
                    cotb = format!("{}{:02}", yb, indexb);
                }
                cota.cmp(&cotb).into()
            }));
        } else {
            sorter_c = Some(gtk::CustomSorter::new(move |a, b| {
                let sa = a.downcast_ref::<gtk::StringList>().unwrap().string(i.try_into().unwrap()).unwrap()
                    .to_lowercase()
                    .replace('á', "a")
                    .replace('é', "e")
                    .replace('í', "i")
                    .replace('ó', "o")
                    .replace('ú', "u")
                    .replace('ü', "u")
                    .replace('ñ', "n~")
                    .replace('ç', "c~");
                let sb = b.downcast_ref::<gtk::StringList>().unwrap().string(i.try_into().unwrap()).unwrap()
                    .to_lowercase()
                    .replace('á', "a")
                    .replace('é', "e")
                    .replace('í', "i")
                    .replace('ó', "o")
                    .replace('ú', "u")
                    .replace('ü', "u")
                    .replace('ñ', "n~")
                    .replace('ç', "c~");
                sa.cmp(&sb).into()
            }));
        }
        col.set_sorter(sorter_c.as_ref());
        if i == j { columnview.sort_by_column(Some(&col), sortdir) }
        if i == 2 { col.set_expand(true) } // Nombre
    }
    columnview.sorter().unwrap().connect_changed(move |sorter, _| {
        let cv_sorter = sorter.downcast_ref::<gtk::ColumnViewSorter>().unwrap();
        let tit = cv_sorter.primary_sort_column().unwrap().title().unwrap();
        let index: i32 = cn.iter().position(|&r| r == tit).unwrap().try_into().unwrap();
        let _ = settings.set_int("columna", index);
        let sort_order = cv_sorter.primary_sort_order();
        if sort_order == gtk::SortType::Ascending {
            let _ = settings.set_boolean("ascendente", true);
        } else {
            let _ = settings.set_boolean("ascendente", false);
        }
    });
}

pub fn crea_lista(store: &gtk::gio::ListStore, conn: &Rc<rusqlite::Connection>) {
    let mut stmt = conn.prepare("SELECT * FROM afiliacion").unwrap();
    let nc = stmt.column_count();
    let mut rows = stmt.query([]).unwrap();
    while let Some(row) = rows.next().unwrap() {
        let mut cc = String::new();
        let m = gtk::StringList::new(&[]);
        for i in 0..nc {
            let s;
            if i != 4 {
                s = row.get(i).unwrap_or_else(|_| "".to_string());
                if i == 0 { cc = s.clone() }
            } else { //Cuota
                let f: f64 = row.get(i).unwrap_or_else(|_| 0.0);
                if f == 0.0 { s = "".to_string() }
                else { s = utils::format_euros(Decimal::from_f64(f).unwrap()) }
            }
            m.append(&s);
        }
        if conn.query_row("SELECT mes FROM cotizaciones WHERE cc=? ORDER BY mes DESC", [&cc], |row_mes| {
                let cot = row_mes.get(0).unwrap_or_else(|_| "".to_string());
                let cot_txt = if cot != "" { utils::mes_cod_a_texto(&cot) } else { cot };
                m.splice(18, 0, &[&cot_txt]);
                Ok(())
            }   
        ).is_err() { 
            m.splice(18, 0, &[""]);
        }
        store.append(&m); // Podría ser más eficiente con un único splice en lugar de un append por cada fila
    }
}

fn crea_menu(maf: &gtk::MenuButton, conn: &Rc<rusqlite::Connection>, nav: &adw::NavigationView, window: &adw::ApplicationWindow, store: &gtk::gio::ListStore, slmodel: &gtk::SortListModel, store_cajas: &gtk::gio::ListStore, cv: &gtk::ColumnView) {
    let ops = ["Cotizar", "Afiliar"];
    let menu = gtk::gio::Menu::new();
    for op in ops {
        let action_name: String = op.chars().filter(|&c| !c.is_whitespace()).collect();
        let action = gtk::gio::SimpleAction::new(&format!("{}_af", action_name), None);
        window.add_action(&action);
        menu.append(Some(&op), Some(&format!("win.{}_af", action_name)));
        if op == "Cotizar" {
            let conn_clone = conn.clone();
            action.connect_activate(glib::clone!(
                #[weak]
                nav,
                #[weak]
                window,
                #[weak]
                store,
                #[weak]
                store_cajas,
                move |_, _| {
                    if nav.visible_page().unwrap().title() == "Volver" {
                        pages_af::cotizar(&conn_clone, &nav, &window, &store, None, &store_cajas, None);
                    }
                }
            ));
        } else if op == "Afiliar" {
            let conn_clone = conn.clone();
            action.connect_activate(glib::clone!(
                #[weak]
                nav,
                #[weak]
                window,
                #[weak]
                store,
                #[weak]
                cv,
                move |_, _| {
                    if nav.visible_page().unwrap().title() == "Volver" {
                        pages_af::afiliar(&conn_clone, &nav, &window, &store, None, Some(&cv));
                    }
                }
            ));
        }
    }
    let ops = ["Exportar csv", "Importar csv"]; 
    let sub = gtk::gio::Menu::new();   
    for op in ops {
        let action_name: String = op.chars().filter(|&c| !c.is_whitespace()).collect();
        let action = gtk::gio::SimpleAction::new(&format!("{}_af", action_name), None);
        window.add_action(&action);
        sub.append(Some(&op), Some(&format!("win.{}_af", action_name)));
        if op == "Exportar csv" {
            action.connect_activate(glib::clone!(
                #[weak]
                window,
                #[weak]
                slmodel,
                move |_, _| {
                    ie_csvs::exp_af(&window, &slmodel);
                }
            ));
        } else if op == "Importar csv" {
            let conn_clone = conn.clone();
            action.connect_activate(glib::clone!(
                #[weak]
                window,
                #[weak]
                store,
                move |_, _| {
                    ie_csvs::imp_af(&conn_clone, &window, &store);
                }
            ));
        }
    }
    menu.append_submenu(Some("CSV"), &sub);
    let pop = gtk::PopoverMenu::from_model(Some(&menu));
    let cont_maf = adw::ButtonContent::builder()
        .icon_name("document-properties-symbolic")
        .label("Acciones")
        .build();
    maf.set_popover(Some(&pop));
    maf.set_child(Some(&cont_maf));
    maf.add_css_class("raised");
    let app = window.application().unwrap();
    app.set_accels_for_action("win.Afiliar_af", &["<ctrl>n"]);
    app.set_accels_for_action("win.Cotizar_af", &["<ctrl>d"]);
}

fn crea_menu_mostrar(mostrar: &gtk::MenuButton, columnview: &gtk::ColumnView, window: &adw::ApplicationWindow) {
    let settings = gtk::gio::Settings::new("org.cntait.tesoreria.afiliacion.mostrar");
    let menu = gtk::gio::Menu::new();
    for col in &columnview.columns() {
        let binding = col.unwrap();
        let column = binding.downcast_ref::<gtk::ColumnViewColumn>().unwrap();
        let tit = column.title().unwrap();
        let action_name: String = tit.to_lowercase().replace("é", "e").replace("ó", "o").replace("/", "-").chars().filter(|&c| !c.is_whitespace()).collect();
        let vs = settings.boolean(&action_name);
        let action = gtk::gio::SimpleAction::new_stateful(&action_name, None, &vs.to_variant());
        window.add_action(&action);
        if tit != "Fecha baja" {
            menu.append(Some(&tit), Some(&format!("win.{}", action_name)));
            column.set_visible(vs);
        } else {
            column.set_visible(false);
        }
        let settings_clone = settings.clone();
        action.connect_activate(glib::clone!(
            #[weak]
            column,
            move |action, _| {
                let v = !column.is_visible();
                column.set_visible(v);
                action.change_state(&v.to_variant());
                let _ = settings_clone.set_boolean(&action.name(), v);
            }
        ));
    }
    let pop = gtk::PopoverMenu::from_model(Some(&menu));
    pop.set_has_arrow(false);
    pop.set_offset(0, 5);
    mostrar.set_popover(Some(&pop));
}

fn crea_menu_filtros(filtros: &gtk::MenuButton, custfilter: &gtk::CustomFilter) {
    let vbox = gtk::Box::new(gtk::Orientation::Vertical, 0);
    let todos = gtk::CheckButton::with_label("Toda la afiliación");
    todos.set_active(true);
    let custfilter_clone = custfilter.clone();
    todos.connect_toggled(move |_| {
        custfilter_clone.changed(gtk::FilterChange::LessStrict);
    });
    vbox.append(&todos);
    for op in utils::COTIZA {
        let ch = gtk::CheckButton::with_label(op);
        ch.set_group(Some(&todos));
        vbox.append(&ch);
        let custfilter_clone = custfilter.clone();
        ch.connect_toggled(move |_| {
            custfilter_clone.changed(gtk::FilterChange::Different);
        });
    }
    let sep = gtk::Separator::new(gtk::Orientation::Horizontal);
    vbox.append(&sep);
    let hbox = gtk::Box::new(gtk::Orientation::Horizontal, 0);
    hbox.set_margin_top(5);
    hbox.set_margin_bottom(5);
    hbox.set_margin_start(5);
    hbox.set_margin_end(5);
    let lab = gtk::Label::new(Some("Meses deuda"));
    lab.set_halign(gtk::Align::Start);
    lab.set_hexpand(true);
    hbox.append(&lab);
    let sw = gtk::Switch::new();
    hbox.append(&sw);
    vbox.append(&hbox);
    let deud = gtk::SpinButton::with_range(1.0, 6.0, 1.0);
    deud.set_value(6.0);
    deud.set_sensitive(false);
    utils::nokeys(&deud);
    vbox.append(&deud);
    let pop = gtk::Popover::builder()
        .child(&vbox)
        .has_arrow(false)
        .autohide(true)
        .build();
    pop.set_offset(0, 5);
    filtros.set_popover(Some(&pop));
    let custfilter_clone = custfilter.clone();
    let sw_clone = sw.clone();
    deud.connect_value_changed(move |_| {
        sw_clone.grab_focus();
        custfilter_clone.changed(gtk::FilterChange::Different);
    });
    let custfilter_clone = custfilter.clone();
    sw.connect_active_notify(move |sw| {
        if sw.is_active() {
            deud.set_sensitive(true);
            custfilter_clone.changed(gtk::FilterChange::MoreStrict);
        } else {
            deud.set_sensitive(false);
            custfilter_clone.changed(gtk::FilterChange::LessStrict);
        }
    });
    let filtros_clone = filtros.clone();
    pop.connect_map(move |_| {
        filtros_clone.remove_css_class("warning");
    });
    let filtros_clone = filtros.clone();
    pop.connect_closed(move |_| {
        if sw.is_active() || !todos.is_active() {
            filtros_clone.add_css_class("warning");
        }
    });
}

fn filtros(item: &glib::Object, dd: &gtk::DropDown, filtros: &gtk::MenuButton, searchentry: &gtk::SearchEntry) -> bool {
    let stl = item.downcast_ref::<gtk::StringList>().unwrap();
    // Filtro Altas o Bajas
    let fbaja = stl.string(19).unwrap();
    if dd.selected() == 0 && fbaja == "" {  }
    else if dd.selected() == 1 && fbaja != "" {  }
    else { return false; }
    // Filtro forma de cotización
    let mut ch = filtros.popover().unwrap().child().unwrap().first_child().unwrap();
    if !ch.downcast_ref::<gtk::CheckButton>().unwrap().is_active() {
        for op in utils::COTIZA {
            ch = ch.next_sibling().unwrap();
            if ch.downcast_ref::<gtk::CheckButton>().unwrap().is_active() { 
                if stl.string(3).unwrap() != *op { return false; }
                break;
            }
        }
    }
    // Filtro deuda meses
    let binding = filtros.popover().unwrap().child().unwrap().last_child().unwrap();
    let sp = binding.downcast_ref::<gtk::SpinButton>().unwrap();
    if sp.is_sensitive() {
        let ma = stl.string(18).unwrap();
        if ma != "" {
            let mut spl = ma.split_whitespace();
            let m = spl.next().unwrap();
            let month: i32 = utils::MESES.iter().position(|&r| r == m).unwrap().try_into().unwrap();
            let year = spl.next().unwrap().parse::<i32>().unwrap();
            let now = glib::DateTime::now_local().unwrap();
            let month_now = now.month();
            let year_now = now.year();
            let dif = (year_now - year)*12 + month_now - month;
            if dif < sp.value_as_int() { return false; }
        }
    }
    // Filtro búsqueda
    let b = searchentry.text();
    if b != "" {
        let mut aux = false;
        for i in 0..stl.n_items() {
            if stl.string(i).unwrap().to_lowercase().contains(&b.to_lowercase()) {
                aux = true;
                break;
            }
        }
        if !aux { return false; }
    }
    true
}

fn nfilas_nsel(fstore: &gtk::FilterListModel, sel: &gtk::MultiSelection , nf: &gtk::Label, window: &adw::ApplicationWindow) {
    let nitems = fstore.n_items();
    let plural = if nitems == 1 { "" } else { "s" };
    let sel_bitset = sel.selection();
    let sel_nitems = sel_bitset.size();
    let sel_plural = if sel_nitems == 1 { "" } else { "s" };
    if sel_nitems != 0 {
        nf.set_label(&format!("{} fila{} | {} seleccionada{}", nitems, plural, sel_nitems, sel_plural));
    } else {
        nf.set_label(&format!("{} fila{}", nitems, plural));
    }
    let enab = if sel_nitems == 1 { true } else { false };
    window.lookup_action("Editar_af_ctx").unwrap().downcast_ref::<gtk::gio::SimpleAction>().unwrap().set_enabled(enab);
}

fn crea_menu_contextual(conn: &Rc<rusqlite::Connection>, nav: &adw::NavigationView, window: &adw::ApplicationWindow, store: &gtk::gio::ListStore, sel: &gtk::MultiSelection, store_cajas: &gtk::gio::ListStore) -> gtk::PopoverMenu {
    let ops = ["Cotizar", "Editar", "Causar baja", "Reincorporar", "Eliminar"];
    let menu = gtk::gio::Menu::new();
    for op in ops {
        let action_name: String = op.chars().filter(|&c| !c.is_whitespace()).collect();
        let action = gtk::gio::SimpleAction::new(&format!("{}_af_ctx", action_name), None);
        window.add_action(&action);
        if op == ops[0] {
            menu.append(Some(&op), Some(&format!("win.{}_af_ctx", action_name)));
            let conn_clone = conn.clone();
            action.connect_activate(glib::clone!(
                #[weak]
                nav,
                #[weak]
                window,
                #[weak]
                store,
                #[weak]
                sel,
                #[weak]
                store_cajas,
                move |_, _| {
                    pages_af::cotizar(&conn_clone, &nav, &window, &store, Some(&sel), &store_cajas, None);
                }
            ));
        } else if op == ops[1] {
            menu.append(Some(&op), Some(&format!("win.{}_af_ctx", action_name)));
            let conn_clone = conn.clone();
            action.connect_activate(glib::clone!(
                #[weak]
                nav,
                #[weak]
                window,
                #[weak]
                store,
                #[weak]
                sel,
                move |_, _| {
                    let pos = sel.selection().minimum();
                    let binding = sel.item(pos).unwrap();
                    let osl = binding.downcast_ref::<gtk::StringList>();
                    pages_af::afiliar(&conn_clone, &nav, &window, &store, osl, None);
                }
            ));
        } else if op == ops[2] {
            menu.append(Some(&op), Some(&format!("win.{}_af_ctx", action_name)));
            let conn_clone = conn.clone();
            action.connect_activate(glib::clone!(
                #[weak]
                window,
                #[weak]
                store,
                #[weak]
                sel,
                move |_, _| {
                    pages_af::causar_baja(&conn_clone, &window, &store, &sel);
                }
            ));
        } else if op == ops[3] {
            let conn_clone = conn.clone();
            action.connect_activate(glib::clone!(
                #[weak]
                window,
                #[weak]
                store,
                #[weak]
                sel,
                move |_, _| {
                    pages_af::reincorporar(&conn_clone, &window, &store, &sel);
                }
            ));
        } else if op == ops[4] {
            let conn_clone = conn.clone();
            action.connect_activate(glib::clone!(
                #[weak]
                window,
                #[weak]
                store,
                #[weak]
                sel,
                move |_, _| {
                    pages_af::eliminar(&conn_clone, &window, &store, &sel);
                }
            ));
        }
    }
    gtk::PopoverMenu::builder()
        .menu_model(&menu)
        .has_arrow(false)
        .halign(gtk::Align::Start)
        .build()
}
