use adw::prelude::*;
use gtk::glib;
use std::cell::RefCell;
use std::rc::Rc;
use rust_decimal::prelude::*;
use crate::db_utils;
use crate::utils;

const MAX: i32 = 99;
const MIN: i32 = 1;

struct Cotizante {
    sl: gtk::StringList,
    ms: i32,
    mcot: String,
    cuota: Decimal,
    proxcot: String,
    limit_inf: String,
    limit_sup: String,
    del_cot: String,
}

struct CotizanteDel {
    sl: gtk::StringList,
    cot: String,
}

pub fn cotizar(conn: &Rc<rusqlite::Connection>, nav: &adw::NavigationView, window: &adw::ApplicationWindow, store: &gtk::gio::ListStore, osel: Option<&gtk::MultiSelection>, store_cajas: &gtk::gio::ListStore, osl_ca: Option<&gtk::StringList>)
{
    let sel = match osel {
        Some(x) => x,
        None => &gtk::MultiSelection::new(Some(gtk::gio::ListStore::new::<gtk::StringList>())),
    };
    let bitset = sel.selection();
    let mut size = bitset.size();
    let vs: Rc<RefCell<Vec<Cotizante>>> = Rc::new(RefCell::new(Vec::new()));
    let vdel: Rc<RefCell<Vec<CotizanteDel>>> = Rc::new(RefCell::new(Vec::new()));
    let vs_clone = Rc::clone(&vs);
    for i in 0..size {
        let pos = bitset.nth(i as u32);
        let binding = sel.item(pos).unwrap();
        let sl = binding.downcast::<gtk::StringList>().unwrap();
        let cuota = match utils::euros_to_dec(&sl.string(4).unwrap()) {
            Ok(num) => num,
            Err(_)  => Decimal::new(000, 2),
        };
        let mut vs = vs_clone.borrow_mut();
        vs.push(Cotizante{sl: sl.clone(), ms: 1, mcot: sl.string(3).unwrap().to_string(), cuota: cuota, proxcot: utils::suma_mes(&sl.string(18).unwrap()), limit_inf: utils::mes_texto_a_cod(&sl.string(18).unwrap()), limit_sup: "".to_string(), del_cot: "".to_string()});
    }
    let sl_ca = match osl_ca {
        Some(x) => x,
        None => &gtk::StringList::new(&["","","","","","",""]),
    };
    if osl_ca != None {
        let caja_sel = store_cajas.item(0).and_downcast::<gtk::StringList>().unwrap().string(0).unwrap();
        let mut stmt = conn.prepare("SELECT cc, Count(*), mes FROM cotizaciones WHERE caja=?1 AND num=?2 GROUP BY cc").unwrap();
        let mut rows = stmt.query([caja_sel.to_string(), sl_ca.string(0).unwrap().to_string()]).unwrap();
        while let Some(row) = rows.next().unwrap() {
            let cc: String = row.get(0).unwrap_or_else(|_| { "".to_string() });
            let ms: i32 = row.get(1).unwrap();
            let codmes: String = row.get(2).unwrap();
            let pos_store = utils::pos_cc(&store, &cc);
            if pos_store == None { continue; }
            let sl = store.item(pos_store.unwrap()).and_downcast::<gtk::StringList>().unwrap();
            let cuota = match utils::euros_to_dec(&sl.string(4).unwrap()) {
                Ok(num) => num,
                Err(_)  => Decimal::new(000, 2),
            };
            let mut limit_sup = String::new();
            let _ = conn.query_row("SELECT mes FROM cotizaciones WHERE NOT(caja=?1 AND num=?2) AND mes>?3 AND cc=?4 ORDER BY mes ASC", [&caja_sel.to_string(), &sl_ca.string(0).unwrap().to_string(), &codmes, &cc], |row| {
                limit_sup = row.get(0).unwrap();
                Ok(())
            });
            let mut limit_inf = String::new();
            let _ = conn.query_row("SELECT mes FROM cotizaciones WHERE NOT(caja=?1 AND num=?2) AND mes<=?3 AND cc=?4 ORDER BY mes DESC", [&caja_sel.to_string(), &sl_ca.string(0).unwrap().to_string(), &codmes, &cc], |row| {
                limit_inf = row.get(0).unwrap();
                Ok(())
            });
            let del_cot = if limit_sup == "" {utils::mes_cod_a_texto(&limit_inf)} else {"".to_string()};
            let mut vs = vs_clone.borrow_mut();
            vs.push(Cotizante{sl: sl.clone(), ms, mcot: sl.string(3).unwrap().to_string(), cuota, proxcot: utils::mes_cod_a_texto(&codmes), limit_inf, limit_sup, del_cot});
            size += 1;
        }
    }
    let over = adw::ToastOverlay::new();
    let vbox = gtk::Box::new(gtk::Orientation::Vertical, 0);
    let hb = adw::HeaderBar::new();
    vbox.append(&hb);
    let pp = adw::PreferencesPage::new();
    pp.set_vexpand(true);
    
    let conc = adw::EntryRow::builder()
        .title("Concepto")
        .activates_default(true)
        .build();
    let total = gtk::Label::new(Some(&calcula_total(&vs.borrow_mut())));
    total.add_css_class("title-4");
    let caja = adw::ComboRow::builder()
        .title("Caja")
        .build();

    let pg = adw::PreferencesGroup::new();
    let (spinm, lab_spinm, dism_spinm, aum_spinm) = custom_spin_meses(1);
    if osl_ca != None {
        let vs = vs_clone.borrow_mut();
        if vs.len() > 0 {
            let v = vs[0].ms;
            let sp = if v == 1 { "mes" } else { "meses" };
            let fmt = format!("{} {}", v, sp);
            lab_spinm.set_label(&fmt);
            if !vs.iter().all(|item| item.ms == v) {
                lab_spinm.set_visible(false);
            }
        }
    }
    let lab_spinm_clone = lab_spinm.clone();
    let pg_clone = pg.clone();
    let total_clone = total.clone();
    let vs_clone = Rc::clone(&vs);
    aum_spinm.connect_clicked(move |_| {
        lab_spinm_clone.set_visible(true);
        let mut v = lab_spinm_clone.text().split_whitespace().next().unwrap().parse::<i32>().unwrap();
        v = if v < MAX { v + 1 } else { MAX };
        let fmt = format!("{} meses", v);
        lab_spinm_clone.set_label(&fmt);
        put_meses(&pg_clone, &fmt);
        let mut vs = vs_clone.borrow_mut();
        for i in 0..vs.len() {
            vs[i].ms = v;
            if vs[i].limit_sup != "" {
                if !check_proxcot(&vs[i]) {
                    put_wgs(&pg_clone, i as u32, true);
                }
            }
        }
        total_clone.set_label(&calcula_total(&vs));
    });
    let pg_clone = pg.clone();
    let vs_clone = Rc::clone(&vs);
    let lab_spinm_clone = lab_spinm.clone();
    let total_clone = total.clone();
    dism_spinm.connect_clicked(move |_| {
        lab_spinm_clone.set_visible(true);
        let mut v = lab_spinm_clone.text().split_whitespace().next().unwrap().parse::<i32>().unwrap();
        v = if v > MIN { v - 1 } else { MIN };
        let sp = if v == 1 { "mes" } else { "meses" };
        let fmt = format!("{} {}", v, sp);
        lab_spinm_clone.set_label(&fmt);
        put_meses(&pg_clone, &fmt);
        let mut vs = vs_clone.borrow_mut();
        for i in 0..vs.len() { 
            vs[i].ms = v; 
            if vs[i].limit_sup != "" {
                if check_proxcot(&vs[i]) {
                    put_wgs(&pg_clone, i as u32, false);
                }
            }
        }
        total_clone.set_label(&calcula_total(&vs));
    });
    pg.set_header_suffix(Some(&spinm));
    if size == 0 {
        spinm.set_visible(false);
    } else if size == 1 {
        pg.set_title("Cotizante");
        spinm.set_visible(false);
    } else {
        pg.set_title("Cotizantes");
    }
    let vs_clone = Rc::clone(&vs);
    for i in 0..size {
        let vsb = vs_clone.borrow_mut();
        let mut cc = vsb[i as usize].sl.string(0).unwrap().to_string();
        let ape = vsb[i as usize].sl.string(1).unwrap();
        let nom = vsb[i as usize].sl.string(2).unwrap();
        if ape != "" && nom != "" {
            cc.push_str(&format!(" | {}, {}", ape, nom));
        } else if ape != "" || nom != ""  {
            cc.push_str(&format!(" | {}{}", ape, nom));
        }        
        let setup_warning = if vsb[i as usize].mcot == utils::COTIZA[4] || vsb[i as usize].cuota == Decimal::ZERO || vsb[i as usize].proxcot == "" {
            true
        } else {
            false
        };
        let af = crear_af_row(&conn, &vs, &cc, setup_warning, &lab_spinm, vsb[i as usize].ms, &pg, &conc, &total, &window, &vdel, &pp, &sl_ca, &store_cajas);
        pg.add(&af);
    }
    let br = adw::ButtonRow::builder()
        .start_icon_name("list-add-symbolic")
        .title("Añadir cotizante")
        .build();
    let window_clone = window.clone();
    let store_clone = store.clone();
    let pg_clone = pg.clone();
    let over_clone = over.clone();
    let conc_clone = conc.clone();
    let total_clone = total.clone();
    let caja_clone = caja.clone();
    let vs_clone = Rc::clone(&vs);
    let vdel_clone = Rc::clone(&vdel);
    let conn_clone = conn.clone();
    let pp_clone = pp.clone();
    let sl_ca_clone = sl_ca.clone();
    let store_cajas_clone = store_cajas.clone();
    br.connect_activated(move |br| {
        add_af_row(&conn_clone, &br, &vs_clone, &store_clone, &pg_clone, &lab_spinm, &over_clone, &conc_clone, &total_clone, &caja_clone, &window_clone, &vdel_clone, &pp_clone, &sl_ca_clone, &store_cajas_clone);
    });
    pg.add(&br);
    pp.add(&pg);
    
    let pg2 = adw::PreferencesGroup::builder()
        .title("Movimiento")
        .build();
        
    let sl_cajas = if osl_ca == None {
        db_utils::lista_cajas(&conn)
    } else {
        let s = store_cajas.item(0).and_downcast::<gtk::StringList>().unwrap().string(0).unwrap();
        gtk::StringList::new(&[&s])
    };
    caja.set_model(Some(&sl_cajas));
    if size != 0 && osl_ca == None {
        let vs_clone = Rc::clone(&vs);
        let vs = vs_clone.borrow_mut();
        sel_caja(&conn, &vs, &caja);
    }
    pg2.add(&caja);
    if size == 1 && osl_ca == None {
        let vs_clone = Rc::clone(&vs);
        let vs = vs_clone.borrow_mut();
        conc.set_text(&format!("{} cotiza 1 mes", vs[0].sl.string(0).unwrap()));
    } else if osl_ca != None {
        conc.set_text(&sl_ca.string(3).unwrap());
    }
    pg2.add(&conc);
    let init = if osl_ca == None { 
        let now = glib::DateTime::now_local().unwrap();
        format!("{:02}/{:02}/{}" ,now.day_of_month(), now.month(), now.year())
    } else {
        sl_ca.string(1).unwrap().to_string()
    };
    let fech = adw::EntryRow::builder()
        .title("Fecha")
        .tooltip_text("dd/mm/aaaa")
        .activates_default(true)
        .text(init)
        .build();
    utils::menusuf_calendar(&fech, &utils::calendario(&fech, true));
    pg2.add(&fech);
    pp.add(&pg2);

    let pga = adw::PreferencesGroup::new();
    let acept= gtk::Button::builder()
        .halign(gtk::Align::Center)
        .css_classes(["pill", "suggested-action"])
        .label("Aplicar")
        .build();
    pga.add(&acept);
    pp.add(&pga);
    
    vbox.append(&pp);
    let actbar = gtk::ActionBar::new();
    actbar.set_center_widget(Some(&total));
    vbox.append(&actbar);
    over.set_child(Some(&vbox));
    let tit = if osl_ca == None { "Cotizar" } else { "Editar cotización" };
    let page = adw::NavigationPage::new(&over, tit);
    nav.push(&page);
    window.set_default_widget(Some(&acept));
    let store_clone = store.clone();
    let store_cajas_clone = store_cajas.clone();
    let conn_clone = conn.clone();
    let nav_clone = nav.clone();
    let sl_ca_clone = sl_ca.clone();
    acept.connect_clicked(move |_| {
        let toast = adw::Toast::new("");
        let vs_clone = Rc::clone(&vs);
        let mut vs = vs_clone.borrow_mut();
        if vs.len() == 0 {
            toast.set_title("Se debe añadir algún cotizante.");
        } else if conc.text() == ""{
            toast.set_title("Debe existir un concepto.");
            conc.grab_focus();
        } else if caja.selected() == gtk::INVALID_LIST_POSITION {
            toast.set_title("Debe existir una caja.");
        } else if !utils::check_date(&fech.text()) || fech.text() == "" {
            toast.set_title("Debe existir una fecha válida.");
            fech.grab_focus();
        } else {
            let mut err = false;
            for i in 0..vs.len() {
                if vs[i].mcot == utils::COTIZA[4] || vs[i].cuota == Decimal::ZERO || !check_proxcot(&vs[i as usize]) {
                    toast.set_title("Edita los cotizantes con alerta.");
                    err = true;
                    break;
                }
            }
            if !err {
                nav_clone.pop();
                let mut importe = Decimal::new(000, 2);
                for i in 0..vs.len() {
                    importe += vs[i].cuota * Decimal::from(vs[i].ms);
                }
                let fecha_bd = utils::convert_fech(&fech.text().to_string());
                let tx = conn_clone.unchecked_transaction().unwrap();
                let caja_str = caja.selected_item().unwrap().downcast::<gtk::StringObject>().unwrap().string();
                let imp = format!("{:.2} €", importe).replace(".", ",");
                let mut num = if sl_ca_clone.string(0).unwrap() == ""  { 0 } else { sl_ca_clone.string(0).unwrap().parse().unwrap() };
                if tit == "Cotizar" {
                    let query = format!("INSERT INTO '{}' (fecha, concepto, importe, etiquetas) VALUES (?1, ?2, ?3, ?4)", caja_str);
                    tx.execute(&query, [&fecha_bd, &conc.text().to_string(), &importe.to_string(), "Cotización"]).unwrap();
                    num = tx.last_insert_rowid();
                    if caja_str == store_cajas_clone.item(0).and_downcast::<gtk::StringList>().unwrap().string(0).unwrap() {
                        let sl_mov = gtk::StringList::new(&[&num.to_string(), &fech.text(), "", &conc.text(), "Cotización", &imp, ""]);
                        utils::recrea_lista(&store_cajas_clone, &sl_mov);
                    }
                } else {
                    let query = format!("UPDATE '{}' SET fecha=?1, concepto=?2, importe=?3, etiquetas=?4 WHERE num=?5", caja_str);
                    tx.execute(&query, [&fecha_bd, &conc.text().to_string(), &importe.to_string(), "Cotización", &num.to_string()]).unwrap();
                    let sl_mov = gtk::StringList::new(&[&num.to_string(), &fech.text(), "", &conc.text(), "Cotización", &imp, ""]);
                    utils::recrea_lista_upd(&store_cajas_clone, &sl_ca_clone, &sl_mov);
                    tx.execute("DELETE FROM cotizaciones WHERE caja=?1 AND num=?2", [caja_str.to_string(), num.to_string()]).unwrap();
                    let vdel_clone = Rc::clone(&vdel);
                    let vdel = vdel_clone.borrow_mut();
                    for i in 0..vdel.len() {
                        let pos = store_clone.find(&vdel[i].sl).unwrap();
                        let sln = gtk::StringList::new(&[]);
                        for j in 0..vdel[i].sl.n_items() {
                            if j == 18 {
                                sln.append(&vdel[i].cot);
                            } else {
                                sln.append(&vdel[i].sl.string(j).unwrap());
                            }
                        }
                        store_clone.splice(pos, 1, &[sln])
                    }
                }
                for i in 0..vs.len() {
                    let cc = vs[i].sl.string(0).unwrap();
                    for j in 0..vs[i].ms {
                        if j != 0 { vs[i].proxcot = utils::suma_mes(&vs[i].proxcot); }
                        let cod_mes = utils::mes_texto_a_cod(&vs[i].proxcot);
                        tx.execute("INSERT INTO cotizaciones (cc, num, caja, mes) VALUES (?1, ?2, ?3, ?4)", 
                            [&cc.to_string(), &num.to_string(), &caja_str.to_string(), &cod_mes]
                        ).unwrap();
                    }
                    let cuota_ant = match utils::euros_to_dec(&vs[i].sl.string(4).unwrap()) {
                        Ok(num) => num,
                        Err(_)  => Decimal::new(000, 2),
                    };
                    if vs[i].sl.string(3).unwrap() != vs[i].mcot || cuota_ant != vs[i].cuota {
                        tx.execute(
                            "UPDATE afiliacion SET mcotiza = ?1, cuota = ?2
                            WHERE cc = ?3",
                            [&vs[i].mcot, &vs[i].cuota.to_string(), &cc.to_string()],
                        ).unwrap();
                    }
                    let pos = store_clone.find(&vs[i].sl).unwrap();
                    let sln = gtk::StringList::new(&[]);
                    for j in 0..vs[i].sl.n_items() {
                        if j == 3 {
                            sln.append(&vs[i].mcot);
                        }
                        else if j == 4 {
                            let cuota_str = utils::format_euros(vs[i].cuota);
                            sln.append(&cuota_str);
                        }
                        else if j == 18 && vs[i].limit_sup == "" {
                            sln.append(&vs[i].proxcot);
                        } else {
                            sln.append(&vs[i].sl.string(j).unwrap());
                        }
                    }
                    store_clone.splice(pos, 1, &[sln])
                }
                let _ = tx.commit();
                return;
            }
        }
        toast.set_timeout(2);
        over.add_toast(toast);
    });
}

pub fn afiliar(conn: &Rc<rusqlite::Connection>, nav: &adw::NavigationView, window: &adw::ApplicationWindow, store: &gtk::gio::ListStore, osl: Option<&gtk::StringList>, ocv: Option<&gtk::ColumnView>) {
    let sl = match osl {
        Some(x) => x,
        None => &gtk::StringList::new(&["","","","","","","","","","","","","","","","","","","",""]),
    };
    let cv = match ocv {
        Some(x) => x,
        None => &gtk::ColumnView::new(Some(gtk::NoSelection::new(Some(gtk::StringList::new(&[]))))),
    };
    let pos = store.find(sl);
    let cc_ant = sl.string(0).unwrap();
    let cotizado = sl.string(18).unwrap();
    let fechabaja = sl.string(19).unwrap();
    
    let vbox = gtk::Box::new(gtk::Orientation::Vertical, 0);
    let hb = adw::HeaderBar::new();
    vbox.append(&hb);
    let pp = adw::PreferencesPage::new();
    let pg_pers = adw::PreferencesGroup::builder()
        .title("Datos personales")
        .build();
    let nom = adw::EntryRow::builder()
        .title("Nombre")
        .activates_default(true)
        .text(sl.string(2).unwrap())
        .build();
    pg_pers.add(&nom);
    let ape = adw::EntryRow::builder()
        .title("Apellidos")
        .activates_default(true)
        .text(sl.string(1).unwrap())
        .build();
    pg_pers.add(&ape);
    let nac = adw::EntryRow::builder()
        .title("Fecha nacimiento")
        .tooltip_text("dd/mm/aaaa")
        .activates_default(true)
        .text(sl.string(5).unwrap())
        .build();
    utils::menusuf_calendar(&nac, &utils::calendario(&nac, false));
    pg_pers.add(&nac);
    let dir = adw::EntryRow::builder()
        .title("Dirección")
        .activates_default(true)
        .text(sl.string(8).unwrap())
        .build();
    pg_pers.add(&dir);
    let loc = adw::EntryRow::builder()
        .title("Localidad")
        .activates_default(true)
        .text(sl.string(9).unwrap())
        .build();
    autocompletado(&loc, "loc", &conn);
    pg_pers.add(&loc);
    let barr = adw::EntryRow::builder()
        .title("Barrio")
        .activates_default(true)
        .text(sl.string(10).unwrap())
        .build();
    autocompletado(&barr, "barrio", &conn);
    pg_pers.add(&barr);
    let tel = adw::EntryRow::builder()
        .title("Teléfono")
        .activates_default(true)
        .text(sl.string(6).unwrap())
        .build();
    pg_pers.add(&tel);
    let email = adw::EntryRow::builder()
        .title("Correo electrónico")
        .activates_default(true)
        .text(sl.string(7).unwrap())
        .build();
    pg_pers.add(&email);
    let adc = adw::EntryRow::builder()
        .title("Información adicional")
        .activates_default(true)
        .text(sl.string(11).unwrap())
        .build();
    pg_pers.add(&adc);
    pp.add(&pg_pers);
    
    let pg_lab = adw::PreferencesGroup::builder()
        .title("Datos laborales")
        .build();
    let ocu = adw::EntryRow::builder()
        .title("Ocupación")
        .activates_default(true)
        .text(sl.string(12).unwrap())
        .build();
    autocompletado(&ocu, "ocu", &conn);
    pg_lab.add(&ocu);
    let cen = adw::EntryRow::builder()
        .title("Centro de trabajo")
        .activates_default(true)
        .text(sl.string(14).unwrap())
        .build();
    autocompletado(&cen, "centro", &conn);
    pg_lab.add(&cen);
    let dirc = adw::EntryRow::builder()
        .title("Dirección")
        .activates_default(true)
        .text(sl.string(15).unwrap())
        .build();
    autocompletado(&dirc, "dirc", &conn);
    pg_lab.add(&dirc);
    let adcc = adw::EntryRow::builder()
        .title("Información adicional")
        .activates_default(true)
        .text(sl.string(16).unwrap())
        .build();
    autocompletado(&adcc, "obsc", &conn);
    pg_lab.add(&adcc);
    pp.add(&pg_lab);

    let pg_sin = adw::PreferencesGroup::builder()
        .title("Datos sindicales")
        .build();
    let cc = adw::EntryRow::builder()
        .title("Carné confederal")
        .activates_default(true)
        .text(cc_ant.clone())
        .build();
    if osl == None {
        cc.add_css_class("warning");
    }
    cc.connect_changed(glib::clone!(
        #[weak]
        cc,
        move |_| {
            if cc.text_length() != 0 {
                cc.remove_css_class("warning");
                cc.remove_css_class("error");
            } else {
                cc.add_css_class("warning");
            }
        }
    ));
    pg_sin.add(&cc);
    let sec = adw::EntryRow::builder()
        .title("Sección")
        .activates_default(true)
        .text(sl.string(13).unwrap())
        .build();
    autocompletado(&sec, "sec", &conn);
    pg_sin.add(&sec);
    let fech = adw::EntryRow::builder()
        .title("Fecha afiliación")
        .tooltip_text("dd/mm/aaaa")
        .activates_default(true)
        .text(sl.string(17).unwrap())
        .build();
    utils::menusuf_calendar(&fech, &utils::calendario(&fech, false));
    pg_sin.add(&fech);
    let cot = adw::ComboRow::builder()
        .title("Modo cotización")
        .build();
    let store_cot = gtk::StringList::new(&[""]);
    for op in utils::COTIZA {
        store_cot.append(op);
    }
    cot.set_model(Some(&store_cot));
    let cotop = sl.string(3).unwrap();
    if cotop != "" {
        let index: u32 = utils::COTIZA.iter().position(|&r| r == cotop).unwrap().try_into().unwrap();
        cot.set_selected(index+1);
    }
    pg_sin.add(&cot);
    let quote = adw::SpinRow::with_range(0.0, 100.0, 1.0);
    quote.set_title("Cuota mensual (€)");
    quote.set_digits(2);
    if osl == None {
        let settings = gtk::gio::Settings::new("org.cntait.tesoreria");
        let sett_cuota = settings.double("cuota");
        quote.set_value(sett_cuota);        
    } else if sl.string(4).unwrap() != "" {
        let cuota_ant = utils::euros_to_dec(&sl.string(4).unwrap()).unwrap();
        quote.set_value(cuota_ant.to_f64().unwrap());
    }
    pg_sin.add(&quote);
    pp.add(&pg_sin);

    let pga = adw::PreferencesGroup::new();
    let acept= gtk::Button::builder()
        .halign(gtk::Align::Center)
        .css_classes(["pill", "suggested-action"])
        .label("Aplicar")
        .build();
    pga.add(&acept);
    pp.add(&pga);
    
    vbox.append(&pp);
    let over = adw::ToastOverlay::new();
    over.set_child(Some(&vbox));
    let tit = if osl == None { "Afiliar" } else { "Editar" };
    let page = adw::NavigationPage::new(&over, tit);
    nav.push(&page);
    if osl == None { 
        nom.grab_focus();
    }
    window.set_default_widget(Some(&acept));
    let conn_clone = conn.clone();
    let nav_clone = nav.clone();
    let store_clone = store.clone();
    let cv_clone = cv.clone();
    acept.connect_clicked(move |_| {
        let toast = adw::Toast::new("");
        if cc.text() == "" {
            toast.set_title("Debes introducir un carné.");
            cc.remove_css_class("warning");
            cc.add_css_class("error");
            cc.grab_focus();
        } else if !utils::check_date(&nac.text()) {
            toast.set_title("Error fecha nacimiento.");
            nac.grab_focus();
        } else if !utils::check_date(&fech.text()) {
            toast.set_title("Error fecha afiliación.");
            fech.grab_focus();
        } else {
            let cotm = cot.selected_item().unwrap().downcast::<gtk::StringObject>().unwrap().string();
            let r;
            if tit == "Afiliar" {
                r = conn_clone.execute(
                    "INSERT INTO afiliacion (cc, apellidos, nombre, mcotiza, cuota, fnacimiento, telefono, email, dir, loc, barrio, obs, ocu, sec, centro, dirc, obsc, afiliado) 
                    VALUES (?1, ?2, ?3, ?4, ?5, ?6, ?7, ?8, ?9, ?10, ?11, ?12, ?13, ?14, ?15, ?16, ?17, ?18)",
                    [&cc.text().to_string(), &ape.text().to_string(), &nom.text().to_string(), &cotm.to_string(), &quote.value().to_string().replace(",", "."), &nac.text().to_string(), &tel.text().to_string(), &email.text().to_string(), &dir.text().to_string(), &loc.text().to_string(), &barr.text().to_string(), &adc.text().to_string(), &ocu.text().to_string(), &sec.text().to_string(), &cen.text().to_string(), &dirc.text().to_string(), &adcc.text().to_string(), &fech.text().to_string()],
                ).unwrap_or_else(|err| {
                    if err.to_string() != "UNIQUE constraint failed: afiliacion.cc" {
                        return 1000;
                    }
                    0
                });
            } else {
                    r = conn_clone.execute(
                    "UPDATE afiliacion SET cc = ?1, apellidos = ?2, nombre = ?3, mcotiza = ?4, cuota = ?5, fnacimiento = ?6, telefono = ?7, email = ?8, dir = ?9, loc = ?10, barrio = ?11, obs = ?12, ocu = ?13, sec = ?14, centro = ?15, dirc = ?16, obsc = ?17, afiliado = ?18
                    WHERE cc = ?19",
                    [&cc.text().to_string(), &ape.text().to_string(), &nom.text().to_string(), &cotm.to_string(), &quote.value().to_string().replace(",", "."), &nac.text().to_string(), &tel.text().to_string(), &email.text().to_string(), &dir.text().to_string(), &loc.text().to_string(), &barr.text().to_string(), &adc.text().to_string(), &ocu.text().to_string(), &sec.text().to_string(), &cen.text().to_string(), &dirc.text().to_string(), &adcc.text().to_string(), &fech.text().to_string(), &cc_ant.to_string()],
                ).unwrap_or_else(|err| {
                    if err.to_string() != "UNIQUE constraint failed: afiliacion.cc" {
                        return 1000;
                    }
                    0
                });
            }
            if r == 0 {
                toast.set_title("Ya existe ese carné.");
                cc.add_css_class("error");
                cc.grab_focus();
            } else if r == 1000{
                toast.set_title("Error. Comprueba los campos.");
            } else {
                nav_clone.pop();
                let cuota;
                let v = quote.value();
                if v == 0.0 { 
                    cuota  = "".to_string();
                } else {
                    cuota = utils::format_euros(Decimal::from_f64(v).unwrap());
                }
                if tit == "Afiliar" {
                    let m = gtk::StringList::new(&[&cc.text(), &ape.text(), &nom.text(), &cotm, &cuota, &nac.text(), &tel.text(), &email.text(), &dir.text(), &loc.text(), &barr.text(), &adc.text(), &ocu.text(), &sec.text(), &cen.text(), &dirc.text(), &adcc.text(), &fech.text(), "", ""]);
                    store_clone.append(&m);
                    utils::select_key_cv(&cv_clone, &cc.text());
                } else {
                    let m = gtk::StringList::new(&[&cc.text(), &ape.text(), &nom.text(), &cotm, &cuota, &nac.text(), &tel.text(), &email.text(), &dir.text(), &loc.text(), &barr.text(), &adc.text(), &ocu.text(), &sec.text(), &cen.text(), &dirc.text(), &adcc.text(), &fech.text(), &cotizado, &fechabaja]);
                    store_clone.splice(pos.unwrap(), 1, &[m]);
                    if cc.text() != cc_ant {
                        conn_clone.execute( "UPDATE cotizaciones SET cc = ?1 WHERE cc = ?2", [cc.text().to_string(), cc_ant.to_string()]).unwrap();
                    }
                }
                let settings = gtk::gio::Settings::new("org.cntait.tesoreria");
                let _ = settings.set_double("cuota", v);
                return;
            }
        }
        toast.set_timeout(2);
        over.add_toast(toast);
    });
}

pub fn causar_baja(conn: &Rc<rusqlite::Connection>, window: &adw::ApplicationWindow, store: &gtk::gio::ListStore, sel: &gtk::MultiSelection) {
    let mut cuerpo = String::new();
    let bitset = sel.selection();
    let size = bitset.size();
    if size == 1 {
        cuerpo.push_str("Carné: ");
    } else {
        cuerpo.push_str("Carnés: ")
    }
    let mut v = Vec::new();
    for i in 0..size {
        let pos = bitset.nth(i as u32);
        let binding = sel.item(pos).unwrap();
        let sl = binding.downcast_ref::<gtk::StringList>().unwrap();
        let cc = sl.string(0).unwrap();
        v.push(sl.clone());
        cuerpo.push_str(&cc);
        cuerpo.push_str(", ");
    }
    cuerpo.pop();
    cuerpo.pop();
    let ad = adw::AlertDialog::new(Some("Causar baja"), Some(&cuerpo));
    ad.add_responses(&[("canc", "Cancelar"), ("acept", "Aceptar")]);
    ad.set_default_response(Some("acept"));
    ad.set_response_appearance("acept", adw::ResponseAppearance::Destructive);
    let pg = adw::PreferencesGroup::new();
    let now = glib::DateTime::now_local().unwrap();
    let today = format!("{:02}/{:02}/{}", now.day_of_month(), now.month(), now.year());
    let fech = adw::EntryRow::builder()
        .title("Fecha baja")
        .tooltip_text("dd/mm/aaaa")
        .activates_default(true)
        .text(today)
        .build();
    utils::menusuf_calendar(&fech, &utils::calendario(&fech, true)); 
    pg.add(&fech);
    ad.set_extra_child(Some(&pg));
    ad.present(Some(window));
    ad.grab_focus();
    let store_clone = store.clone();
    let fech_clone = fech.clone();
    let conn_clone = conn.clone();
    ad.connect_response(Some("acept"), move |_, _| {
        let fecha = fech_clone.text();
        let tx = conn_clone.unchecked_transaction().unwrap();
        for sl in v.clone() {
            let cc = sl.string(0).unwrap();
            let pos_store = store_clone.find(&sl).unwrap();
            sl.splice(19, 1, &[&fecha]);
            store_clone.splice(pos_store, 1, &[sl]);
            tx.execute(
                    "UPDATE afiliacion SET fechabaja = ?1
                    WHERE cc = ?2",
                    [fecha.to_string(), cc.to_string()],
            ).unwrap();
        }
        let _ = tx.commit();
    });
    fech.connect_changed(move |fech| {
        if utils::check_date(&fech.text()) && fech.text() != "" {
            ad.set_response_enabled("acept", true);
        } else {
            ad.set_response_enabled("acept", false);
        }
    });
}

pub fn reincorporar(conn: &Rc<rusqlite::Connection>, window: &adw::ApplicationWindow, store: &gtk::gio::ListStore, sel: &gtk::MultiSelection) {
    let mut cuerpo = String::new();
    let bitset = sel.selection();
    let size = bitset.size();
    if size == 1 {
        cuerpo.push_str("Carné: ");
    } else {
        cuerpo.push_str("Carnés: ")
    }
    let mut v = Vec::new();
    for i in 0..size {
        let pos = bitset.nth(i as u32);
        let binding = sel.item(pos).unwrap();
        let sl = binding.downcast_ref::<gtk::StringList>().unwrap();
        let cc = sl.string(0).unwrap();
        v.push(sl.clone());
        cuerpo.push_str(&cc);
        cuerpo.push_str(", ");
    }
    cuerpo.pop();
    cuerpo.pop();
    let ad = adw::AlertDialog::new(Some("Reincorporar"), Some(&cuerpo));
    ad.add_responses(&[("canc", "Cancelar"), ("acept", "Aceptar")]);
    ad.set_default_response(Some("acept"));
    ad.set_response_appearance("acept", adw::ResponseAppearance::Suggested);
    ad.present(Some(window));
    let store_clone = store.clone();
    let conn_clone = conn.clone();
    ad.connect_response(Some("acept"), move |_, _| {
        let tx = conn_clone.unchecked_transaction().unwrap();
        for sl in v.clone() {
            let cc = sl.string(0).unwrap();
            let pos_store = store_clone.find(&sl).unwrap();
            sl.splice(19, 1, &[""]);
            store_clone.splice(pos_store, 1, &[sl]);
            tx.execute(
                    "UPDATE afiliacion SET fechabaja = ''
                    WHERE cc = ?1",
                    [cc.to_string()],
            ).unwrap();
        }
        let _ = tx.commit();
    });
}

pub fn eliminar(conn: &Rc<rusqlite::Connection>, window: &adw::ApplicationWindow, store: &gtk::gio::ListStore, sel: &gtk::MultiSelection) {
    let mut cuerpo = String::new();
    let bitset = sel.selection();
    let size = bitset.size();
    if size == 1 {
        cuerpo.push_str("Carné: ");
    } else {
        cuerpo.push_str("Carnés: ")
    }
    let mut v = Vec::new();
    for i in 0..size {
        let pos = bitset.nth(i as u32);
        let binding = sel.item(pos).unwrap();
        let sl = binding.downcast_ref::<gtk::StringList>().unwrap();
        let cc = sl.string(0).unwrap();
        v.push(sl.clone());
        cuerpo.push_str(&cc);
        cuerpo.push_str(", ");
    }
    cuerpo.pop();
    cuerpo.pop();
    let ad = adw::AlertDialog::new(Some("Eliminar"), Some(&cuerpo));
    ad.add_responses(&[("canc", "Cancelar"), ("acept", "Aceptar")]);
    ad.set_default_response(Some("acept"));
    ad.set_response_appearance("acept", adw::ResponseAppearance::Destructive);
    ad.present(Some(window));
    let store_clone = store.clone();
    let conn_clone = conn.clone();
    ad.connect_response(Some("acept"), move |_, _| {
        let tx = conn_clone.unchecked_transaction().unwrap();
        for sl in v.clone() {
            let cc = sl.string(0).unwrap();
            let pos_store = store_clone.find(&sl).unwrap();
            store_clone.remove(pos_store);
            tx.execute(
                    "DELETE FROM afiliacion
                    WHERE cc = ?1",
                    [cc.to_string()],
            ).unwrap();
        }
        let _ = tx.commit();
    });
}

pub fn cotizaciones(conn: &Rc<rusqlite::Connection>, nav: &adw::NavigationView, store: &gtk::gio::ListStore) {
    let vbox = gtk::Box::new(gtk::Orientation::Vertical, 10);
    let hb = adw::HeaderBar::new();
    vbox.append(&hb);
    let stack = adw::ViewStack::new();
    stack.set_margin_start(10);
    stack.set_margin_end(10);
    let switcher = adw::ViewSwitcher::builder()
        .stack(&stack)
        .policy(adw::ViewSwitcherPolicy::Wide)
        .halign(gtk::Align::Center)
        .build();
    vbox.append(&switcher);
    vbox.append(&stack);
    let factory = gtk::SignalListItemFactory::new();
    factory.connect_setup(move |_, item| {
        let item = item.downcast_ref::<gtk::ListItem>().unwrap();
        let tit = gtk::Label::new(None); //builder
        tit.add_css_class("caption");
        tit.add_css_class("dim-label");
        let stit = gtk::Label::new(None);
        let suf = gtk::Label::new(None);
        tit.set_halign(gtk::Align::Start);
        stit.set_halign(gtk::Align::Start);
        let mbox = gtk::Box::new(gtk::Orientation::Horizontal, 0);
        let vbox = gtk::Box::new(gtk::Orientation::Vertical, 0);
        vbox.set_hexpand(true);
        vbox.append(&tit);
        vbox.append(&stit);
        mbox.append(&vbox);
        mbox.append(&suf);
        item.set_child(Some(&mbox));
    });
    factory.connect_bind(move |_, item| {
        let item = item.downcast_ref::<gtk::ListItem>().unwrap();
        let mbox = item.child().and_downcast::<gtk::Box>().unwrap();
        let vbox = mbox.first_child().and_downcast::<gtk::Box>().unwrap();
        let tit = vbox.first_child().and_downcast::<gtk::Label>().unwrap();
        let stit = vbox.last_child().and_downcast::<gtk::Label>().unwrap();
        let suf = mbox.last_child().and_downcast::<gtk::Label>().unwrap();
        let strl = item.item().and_downcast::<gtk::StringList>().unwrap();
        let tits = strl.string(0).unwrap();
        let stits = strl.string(1).unwrap();
        let sufs = strl.string(2).unwrap();
        tit.set_label(&tits);
        stit.set_label(&stits);
        suf.set_label(&sufs);
    });
    
    let vboxp = gtk::Box::new(gtk::Orientation::Vertical, 0);
    let (slm, exp) = slm_exp_afiliados(&store);
    let combo = gtk::DropDown::builder()
        .model(&slm)
        .expression(&exp)
        .enable_search(true)
        .search_match_mode(gtk::StringFilterMatchMode::Substring)
        .build();
    let year = gtk::SpinButton::with_range(1910.0, 2999.0, 1.0);
    let now = glib::DateTime::now_local().unwrap();
    year.set_value(now.year() as f64);
    utils::nokeys(&year);
    let hbox = utils::crea_hb("Cotizante:", &combo, "Año:", &year);
    vboxp.append(&hbox);
    let storev = gtk::gio::ListStore::new::<gtk::StringList>();
    let cuotas = cotizaciones_comun(&storev, &factory, &vboxp);
    crea_cotizaciones_afiliad(&conn, &combo, &year, &storev, &cuotas);
    let year_clone = year.clone();
    let conn_clone = conn.clone();
    let storev_clone = storev.clone();
    let cuotas_clone = cuotas.clone();
    combo.connect_selected_notify(move |combo| {
        crea_cotizaciones_afiliad(&conn_clone, &combo, &year_clone, &storev_clone, &cuotas_clone);
    });
    let combo_clone = combo.clone();
    let conn_clone = conn.clone();
    let storev_clone = storev.clone();
    let cuotas_clone = cuotas.clone();
    year.connect_value_notify(move |year|{
        combo_clone.grab_focus();
        crea_cotizaciones_afiliad(&conn_clone, &combo_clone, &year, &storev_clone, &cuotas_clone);
    });
    stack.add_titled_with_icon(&vboxp, None, "Cotizante", "avatar-default-symbolic");

    let vboxp = gtk::Box::new(gtk::Orientation::Vertical, 0);
    let sl = gtk::StringList::new(&utils::MESES[1..]);
    let mes = gtk::DropDown::builder()
        .model(&sl)
        .build();
    let mes_ant = now.add_months(-1).unwrap();
    mes.set_selected((mes_ant.month() -1)  as u32);
    let year = gtk::SpinButton::with_range(1910.0, 2999.0, 1.0);
    year.set_value(mes_ant.year() as f64);
    utils::nokeys(&year);
    let hbox = utils::crea_hb("Mes:", &mes, "Año:", &year);
    vboxp.append(&hbox);
    let storev = gtk::gio::ListStore::new::<gtk::StringList>();
    let cuotas = cotizaciones_comun(&storev, &factory, &vboxp);
    crea_cotizaciones_mes(&conn, &mes, &year, &storev, &cuotas);
    let year_clone = year.clone();
    let conn_clone = conn.clone();
    let storev_clone = storev.clone();
    let cuotas_clone = cuotas.clone();
    mes.connect_selected_notify(move |mes| {
        crea_cotizaciones_mes(&conn_clone, &mes, &year_clone, &storev_clone, &cuotas_clone);
    });
    let mes_clone = mes.clone();
    let conn_clone = conn.clone();
    let storev_clone = storev.clone();
    let cuotas_clone = cuotas.clone();
    year.connect_value_notify(move |year|{
        mes_clone.grab_focus();
        crea_cotizaciones_mes(&conn_clone, &mes_clone, &year, &storev_clone, &cuotas_clone);
    });
    stack.add_titled_with_icon(&vboxp, None, "Mes", "user-bookmarks-symbolic");
    
    let vboxp = gtk::Box::new(gtk::Orientation::Vertical, 0);
    let desde_str = format!("01/{:02}/{}", mes_ant.month(), mes_ant.year());
    let desde = utils::entry_calendar(&desde_str);
    let mut j = glib::Date::from_dmy(31, glib::DateMonth::December, (mes_ant.year() -1) as u16).unwrap();
    let _ = j.add_days(mes_ant.day_of_year() as u32);
    let last = glib::Date::days_in_month(j.month(), j.year());
    let hasta_str = format!("{}/{:02}/{}", last, mes_ant.month(), j.year());
    let hasta = utils::entry_calendar(&hasta_str);
    let hbox = utils::crea_hb("Desde:", &desde, "Hasta:", &hasta);
    vboxp.append(&hbox);
    let storev = gtk::gio::ListStore::new::<gtk::StringList>();
    let cuotas = cotizaciones_comun(&storev, &factory, &vboxp);
    crea_cotizaciones_fechas(&conn, &desde, &hasta, &storev, &cuotas);
    let conn_clone = conn.clone();
    let hasta_clone = hasta.clone();
    let storev_clone = storev.clone();
    let cuotas_clone = cuotas.clone();
    desde.connect_changed(move |desde| {
        crea_cotizaciones_fechas(&conn_clone, &desde, &hasta_clone, &storev_clone, &cuotas_clone);
    });
    let conn_clone = conn.clone();
    let desde_clone = desde.clone();
    let storev_clone = storev.clone();
    let cuotas_clone = cuotas.clone();
    hasta.connect_changed(move |hasta| {
        crea_cotizaciones_fechas(&conn_clone, &desde_clone, &hasta, &storev_clone, &cuotas_clone);
    });
    stack.add_titled_with_icon(&vboxp, None, "Fechas", "x-office-calendar-symbolic");
    
    let vboxp = gtk::Box::new(gtk::Orientation::Vertical, 0);
    let hbox = gtk::Box::new(gtk::Orientation::Horizontal, 1);
    hbox.set_halign(gtk::Align::Center);
    let lab = gtk::Label::new(Some("Año:"));
    hbox.append(&lab);
    let year = gtk::SpinButton::with_range(1910.0, 2999.0, 1.0);
    year.set_value(now.year() as f64);
    utils::nokeys(&year);
    hbox.append(&year);
    vboxp.append(&hbox);
    let vboxc = gtk::Box::new(gtk::Orientation::Vertical, 40);
    let clamp = adw::Clamp::new();
    clamp.set_child(Some(&vboxc));
    let sc = gtk::ScrolledWindow::new();
    sc.set_margin_top(10);
    sc.set_vexpand(true);
    sc.set_child(Some(&clamp));
    vboxp.append(&sc); 
    let pg = adw::PreferencesGroup::new();
    pg.set_title("Meses cotizados");
    let fb1 = gtk::FlowBox::new();
    fb1.set_selection_mode(gtk::SelectionMode::None);
    fb1.set_max_children_per_line(4);
    fb1.set_column_spacing(15);
    pg.add(&fb1);
    vboxc.append(&pg);
    let pg = adw::PreferencesGroup::new();
    pg.set_title("Cuotas cobradas");
    let fb2 = gtk::FlowBox::new();
    fb2.set_selection_mode(gtk::SelectionMode::None);
    fb2.set_max_children_per_line(4);
    fb2.set_column_spacing(15);
    pg.add(&fb2);
    vboxc.append(&pg);
    crea_cotizaciones_estadillos(&conn, &now.year().to_string(), &fb1, &fb2);
    let conn_clone = conn.clone();
    year.connect_value_notify(move |year|{
        fb1.child_at_index(0).unwrap().grab_focus();
        crea_cotizaciones_estadillos(&conn_clone, &year.value().to_string(), &fb1, &fb2);
    });
    stack.add_titled_with_icon(&vboxp, None, "Estadillos", "view-grid-symbolic");
    
    let page = adw::NavigationPage::new(&vbox, "Cotizaciones");
    nav.push(&page);
}

fn cotizaciones_comun(store: &gtk::gio::ListStore, factory: &gtk::SignalListItemFactory, vbox: &gtk::Box) -> gtk::Label {
    let sel = gtk::SingleSelection::new(Some(store.clone()));
    sel.set_autoselect(false);
    sel.set_can_unselect(true);
    let lv = gtk::ListView::new(Some(sel), Some(factory.clone()));
    lv.set_show_separators(true);
    let clamp = adw::Clamp::new();
    clamp.set_child(Some(&lv));
    let sc = gtk::ScrolledWindow::new();
    sc.set_margin_top(10);
    sc.set_vexpand(true);
    sc.set_child(Some(&clamp));
    vbox.append(&sc);
    let cuotas = gtk::Label::new(None);
    cuotas.add_css_class("title-4");
    let actbar = gtk::ActionBar::new();
    actbar.set_center_widget(Some(&cuotas));
    vbox.append(&actbar);
    cuotas
}

fn autocompletado(ent: &adw::EntryRow, col: &str, conn: &Rc<rusqlite::Connection>) {
    let pop = gtk::Popover::builder()
        .has_arrow(false)
        .autohide(false)
        .halign(gtk::Align::Start)
        .css_classes(["menu"])
        .build();
    pop.set_parent(ent);
    let coln = col.to_string();
    let conn_clone = conn.clone();
    ent.connect_changed(glib::clone!(
        #[weak]
        ent,
        #[weak]
        pop,
        move |_| {
            autocomp_aux(&ent, &coln, &pop, &conn_clone);
        }
    ));
    let eventfocus = gtk::EventControllerFocus::new();
    ent.add_controller(eventfocus.clone());
    eventfocus.connect_leave(glib::clone!(
        #[weak]
        pop,
        move |_| {
            pop.popdown();
        }
    ));
}

fn autocomp_aux(ent: &adw::EntryRow, col: &str, pop: &gtk::Popover, conn: &Rc<rusqlite::Connection>) {
    let text_nc = ent.text_length();
    if text_nc == 0 {
        pop.popdown();
        return;
    }
    let text = ent.text().to_lowercase();    
    let mut stmt = conn.prepare(&format!("SELECT DISTINCT {} FROM afiliacion", &col)).unwrap();
    let mut rows = stmt.query([]).unwrap();
    let mut v = Vec::new();
    while let Some(row) = rows.next().unwrap() {
        let m: String = row.get(0).unwrap_or_else(|_| { "".to_string() });
        v.push(m);
    }
    let lb = gtk::ListBox::new();
    let mut cnt = 0;
    for i in 0..v.len() {
        let mut cnt_chs = 0;
        let mut text_chs = text.chars();
        for c in v[i].to_lowercase().chars() {
            if text_chs.next() != Some(c) {
                break;
            }
            cnt_chs += 1;
        }
        if cnt_chs == text_nc {
            cnt += 1;
            let lbr = gtk::ListBoxRow::new();
            lb.append(&lbr);
            let lab = gtk::Label::new(Some(&v[i]));
            lab.set_halign(gtk::Align::Start);
            lbr.set_child(Some(&lab));
        }
        if cnt > 9 {
            break;
        }
    }
    if cnt > 0 {
        pop.set_child(Some(&lb));
        pop.popup();
    } else {
        pop.popdown();
    }
    lb.connect_row_activated(glib::clone!(
        #[weak]
        ent,
        #[weak]
        pop,
        #[weak]
        lb,
        move |_,_| {
            let row = lb.selected_row().unwrap();
            let txt = row.child().unwrap().downcast::<gtk::Label>().unwrap().text();
            ent.set_text(&txt);
            ent.set_position(-1);
            //ent.grab_focus_without_selecting();
            pop.popdown();
        }
    ));
}

fn custom_spin_meses(init: i32) -> (gtk::Box, gtk::Label, gtk::Button, gtk::Button) {
    let hbox = gtk::Box::new(gtk::Orientation::Horizontal, 0);
    let sp = if init == 1 { "mes" } else { "meses" };
    let label = gtk::Label::new(Some(&format!("{} {}", init, sp)));
    hbox.append(&label);
    let dism = gtk::Button::builder()
        .icon_name("value-decrease-symbolic")
        .valign(gtk::Align::Center)
        .css_classes(["flat", "bnopad"])
        .tooltip_text("Disminuir")
        .build();
    hbox.append(&dism);
    let aum = gtk::Button::builder()
        .icon_name("value-increase-symbolic")
        .valign(gtk::Align::Center)
        .css_classes(["flat", "bnopad"])
        .tooltip_text("Aumentar")
        .build();
    hbox.append(&aum);
    (hbox, label, dism, aum)
}

fn put_meses (pg: &adw::PreferencesGroup, texto: &str) {
    let rows = pg.first_child().unwrap().last_child().unwrap().first_child().unwrap().observe_children();
    let n = rows.n_items();
    for i in 0..n-1 {
        rows.item(i).and_downcast::<adw::ActionRow>().unwrap()
            .first_child().unwrap()
            .last_child().unwrap()
            .first_child().unwrap()
            .first_child().unwrap()
            .first_child().and_downcast::<gtk::Label>().unwrap()
            .set_label(texto); 
    }
}

fn put_wgs (pg: &adw::PreferencesGroup, i: u32, w: bool) {
    let rows = pg.first_child().unwrap().last_child().unwrap().first_child().unwrap().observe_children();
    let bt = rows.item(i).and_downcast::<adw::ActionRow>().unwrap()
        .first_child().unwrap()
        .last_child().unwrap()
        .first_child().unwrap()
        .observe_children().item(1)
        .and_downcast::<gtk::Button>().unwrap();
    if w {
        bt.add_css_class("warning");
    } else {
        bt.remove_css_class("warning");
    }
}

fn crear_af_row(conn: &Rc<rusqlite::Connection>, vs: &Rc<RefCell<Vec<Cotizante>>>, txt: &str, warning: bool, lab_spinm: &gtk::Label, spin_init: i32, pg: &adw::PreferencesGroup, conc: &adw::EntryRow, total: &gtk::Label, window: &adw::ApplicationWindow, vdel: &Rc<RefCell<Vec<CotizanteDel>>>, pp: &adw::PreferencesPage, sl_ca: &gtk::StringList, store_cajas: &gtk::gio::ListStore) -> adw::ActionRow {
    let af = adw::ActionRow::builder()
        .title("Carné | Apellidos, Nombre")
        .subtitle(txt)
        .css_classes(["property"])
        .build();
    let hbox = gtk::Box::new(gtk::Orientation::Horizontal, 0);
    let (spin, lab_spin, dism_spin, aum_spin) = custom_spin_meses(spin_init);
    hbox.append(&spin);
    let setup = gtk::Button::builder()
        .margin_start(15)
        .icon_name("document-properties-symbolic")
        .valign(gtk::Align::Center)
        .css_classes(["flat"])
        .tooltip_text("Editar")
        .build();
    if warning { setup.add_css_class("warning"); }
    let window_clone = window.clone();
    let total_clone = total.clone();
    let af_clone = af.clone();
    let conn_clone = conn.clone();
    let sl_ca_clone = sl_ca.clone();
    let store_cajas_clone = store_cajas.clone();
    let vs_clone = Rc::clone(&vs);
    let txt = txt.to_string();
    setup.connect_clicked(move |setup| {
        let ad = adw::AlertDialog::new(Some("Editar cotizante"), Some(&txt));
        ad.add_responses(&[("canc", "Cancelar"), ("acept", "Aceptar")]);
        ad.set_default_response(Some("acept"));
        ad.set_response_appearance("acept", adw::ResponseAppearance::Suggested);
        ad.set_width_request(387);
        let pg = adw::PreferencesGroup::new();
        let i = af_clone.index();
        let vs = vs_clone.borrow_mut();
        if vs[i as usize].mcot == utils::COTIZA[4] {
            let cot = adw::ComboRow::builder()
                .title("Modo cotización")
                .build();
            let store_cot = gtk::StringList::new(&[""]);
            for op in utils::COTIZA {
                store_cot.append(op);
            }
            cot.set_model(Some(&store_cot));
            cot.set_selected(4);
            pg.add(&cot);
        }
        let quote = adw::SpinRow::with_range(0.0, 100.0, 1.0);
        quote.set_title("Cuota mensual (€)");
        quote.set_digits(2);
        quote.set_value(vs[i as usize].cuota.to_f64().unwrap());
        pg.add(&quote);
        let prox = adw::ActionRow::builder()
            .title("Próxima cotización")
            .build();
        let hbox = gtk::Box::new(gtk::Orientation::Horizontal, 5);
        hbox.set_valign(gtk::Align::Center);
        let dmonth = gtk::DropDown::from_strings(utils::MESES);
        hbox.append(&dmonth);
        let dyear = gtk::SpinButton::with_range(1910.0, 2999.0, 1.0);
        hbox.append(&dyear);
        let ma = &vs[i as usize].proxcot;
        if ma != "" {
            let mut iter = ma.split_whitespace();
            let mes = iter.next().unwrap();
            let mes_pos = utils::MESES.iter().position(|&r| r == mes).unwrap();
            dmonth.set_selected(mes_pos as u32);
            let anyo = iter.next().unwrap().parse::<f64>().unwrap();
            dyear.set_value(anyo);
        } else {
            let now = glib::DateTime::now_local().unwrap();
            dyear.set_value(now.year() as f64);
        }
        prox.add_suffix(&hbox);
        pg.add(&prox);
        ad.set_extra_child(Some(&pg));
        ad.present(Some(&window_clone));
        ad.grab_focus();
        let vs_clone = Rc::clone(&vs_clone);
        let setup_clone = setup.clone();
        let total_clone = total_clone.clone();
        let conn_clone = conn_clone.clone();
        let sl_ca_clone = sl_ca_clone.clone();
        let store_cajas_clone = store_cajas_clone.clone();
        ad.connect_response(Some("acept"), move |_, _| {
            let mut quit_warn = 0;
            let mut vs = vs_clone.borrow_mut();
            let lb = prox.parent().unwrap();
            let n = lb.observe_children().n_items();
            if n == 3 {
                let mcot = lb.first_child().and_downcast::<adw::ComboRow>().unwrap().selected_item().unwrap().downcast::<gtk::StringObject>().unwrap().string();
                if mcot != utils::COTIZA[4]  {
                    vs[i as usize].mcot = mcot.to_string();
                    quit_warn += 1;
                }
            }
            let qv = quote.value();
            vs[i as usize].cuota = Decimal::from_f64(qv).unwrap();
            if qv != 0.0 { quit_warn += 1; }
            if dmonth.selected() != 0 {
                let m = dmonth.selected_item().unwrap().downcast::<gtk::StringObject>().unwrap().string();
                let y = dyear.value_as_int();
                vs[i as usize].proxcot = format!("{} {}", m, y);
                let mut limit_sup = String::new();
                let mut limit_inf = String::new();
                if sl_ca_clone.string(0).unwrap().to_string() == "" {
                    let _ = conn_clone.query_row("SELECT mes FROM cotizaciones WHERE mes>?1 AND cc=?2 ORDER BY mes ASC", [utils::mes_texto_a_cod(&vs[i as usize].proxcot), vs[i as usize].sl.string(0).unwrap().to_string()], |row| {
                        limit_sup = row.get(0).unwrap();
                        Ok(())
                    });
                    let _ = conn_clone.query_row("SELECT mes FROM cotizaciones WHERE  mes<=?1 AND cc=?2 ORDER BY mes DESC", [utils::mes_texto_a_cod(&vs[i as usize].proxcot), vs[i as usize].sl.string(0).unwrap().to_string()], |row| {
                        limit_inf = row.get(0).unwrap();
                        Ok(())
                    });
                } else {
                    let caja_sel = store_cajas_clone.item(0).and_downcast::<gtk::StringList>().unwrap().string(0).unwrap();
                    let _ = conn_clone.query_row("SELECT mes FROM cotizaciones WHERE NOT(caja=?1 AND num=?2) AND mes>?3 AND cc=?4 ORDER BY mes ASC", [caja_sel.to_string(), sl_ca_clone.string(0).unwrap().to_string(), utils::mes_texto_a_cod(&vs[i as usize].proxcot), vs[i as usize].sl.string(0).unwrap().to_string()], |row| {
                        limit_sup = row.get(0).unwrap();
                        Ok(())
                    });
                    let _ = conn_clone.query_row("SELECT mes FROM cotizaciones WHERE NOT(caja=?1 AND num=?2) AND mes<=?3 AND cc=?4 ORDER BY mes DESC", [caja_sel.to_string(), sl_ca_clone.string(0).unwrap().to_string(), utils::mes_texto_a_cod(&vs[i as usize].proxcot), vs[i as usize].sl.string(0).unwrap().to_string()], |row| {
                        limit_inf = row.get(0).unwrap();
                        Ok(())
                    });
                }
                vs[i as usize].limit_sup = limit_sup;
                vs[i as usize].limit_inf = limit_inf;
                if check_proxcot(&vs[i as usize]) { quit_warn += 1; }
            } else { vs[i as usize].proxcot = "".to_string() }
            if quit_warn == n {
                setup_clone.remove_css_class("warning");
            } else {
                setup_clone.add_css_class("warning");
            }
            total_clone.set_label(&calcula_total(&vs));
        });
    });
    hbox.append(&setup);
    let del = gtk::Button::builder()
        .icon_name("edit-delete-symbolic")
        .valign(gtk::Align::Center)
        .css_classes(["flat"])
        .tooltip_text("Eliminar")
        .build();
    let vs_clone = Rc::clone(&vs);
    let vdel_clone = Rc::clone(&vdel);
    del.connect_clicked(glib::clone!(
        #[weak]
        pg,
        #[weak]
        af,
        #[weak]
        lab_spinm,
        #[weak]
        conc,
        #[weak]
        total,
        #[weak]
        pp,
        move |_| {
            let sc = pp.first_child().and_downcast::<gtk::ScrolledWindow>().unwrap();
            let vadj = sc.vadjustment();
            let val = vadj.value();
            let i = af.index();
            pg.remove(&af);
            let mut vs = vs_clone.borrow_mut();
            if vs[i as usize].del_cot != ""  {
                let mut vdel = vdel_clone.borrow_mut();
                vdel.push(CotizanteDel{sl: vs[i as usize].sl.clone(), cot: vs[i as usize].del_cot.clone()});
            }
            vs.remove(i as usize);
            total.set_label(&calcula_total(&vs));
            let n = vs.len();
            if n == 0 {
                pg.set_title("");
                conc.set_text("");
            } else if n == 1 {
                pg.set_title("Cotizante");
                pg.header_suffix().unwrap().set_visible(false);
                let ms_str = if vs[0].ms == 1 { "mes" } else { "meses" };
                lab_spinm.set_label(&format!("{} {}", vs[0].ms, ms_str));
                conc.set_text(&format!("{} cotiza {} {}", vs[0].sl.string(0).unwrap(), vs[0].ms, ms_str));
            }
            glib::idle_add_local_once(move|| {
                vadj.set_value(val);
            });
        }
    ));
    hbox.append(&del);
    af.add_suffix(&hbox);
    let lab_spin_clone = lab_spin.clone();
    let vs_clone = Rc::clone(&vs);
    let lab_spinm_clone = lab_spinm.clone();
    let af_clone = af.clone();
    let conc_clone = conc.clone();
    let total_clone = total.clone();
    let setup_clone = setup.clone();
    aum_spin.connect_clicked(move |_| {
        let v = lab_spin_clone.text().split_whitespace().next().unwrap().parse::<i32>().unwrap();
        if v < MAX {
            lab_spin_clone.set_label(&format!("{} meses", v+1));
            lab_spinm_clone.set_label(&lab_spin_clone.label());
            let i = af_clone.index();
            let mut vs = vs_clone.borrow_mut();
            vs[i as usize].ms = v+1;
            total_clone.set_label(&calcula_total(&vs));
            if vs.len() > 1 { 
                let first = vs[0].ms;
                if vs.iter().all(|item| item.ms == first) {
                    lab_spinm_clone.set_visible(true);
                } else {
                    lab_spinm_clone.set_visible(false);
                }
            } else {
                let ms_str = if vs[0].ms == 1 { "mes" } else { "meses" };
                conc_clone.set_text(&format!("{} cotiza {} {}", vs[0].sl.string(0).unwrap(), vs[0].ms, ms_str));
            }
            if vs[i as usize].limit_sup != "" {
                if !check_proxcot(&vs[i as usize]) {
                    setup_clone.add_css_class("warning");
                }
            }
        }
    });
    let vs_clone = Rc::clone(&vs);
    let lab_spinm_clone = lab_spinm.clone();
    let af_clone = af.clone();
    let conc_clone = conc.clone();
    let total_clone = total.clone();
    dism_spin.connect_clicked(move |_| {
        let v = lab_spin.text().split_whitespace().next().unwrap().parse::<i32>().unwrap();
        if v > MIN {
            let sp = if v == 2 { "mes" } else { "meses" };
            lab_spin.set_label(&format!("{} {}", v-1, sp));
            lab_spinm_clone.set_label(&lab_spin.label());
            let i = af_clone.index();
            let mut vs = vs_clone.borrow_mut();
            vs[i as usize].ms = v-1;
            total_clone.set_label(&calcula_total(&vs));
            if vs.len() > 1 {
                let first = vs[0].ms;
                if vs.iter().all(|item| item.ms == first) {
                    lab_spinm_clone.set_visible(true);
                } else {
                    lab_spinm_clone.set_visible(false);
                }
            } else {
                let ms_str = if vs[0].ms == 1 { "mes" } else { "meses" };
                conc_clone.set_text(&format!("{} cotiza {} {}", vs[0].sl.string(0).unwrap(), vs[0].ms, ms_str));
            }
            if vs[i as usize].limit_sup != "" {
                if check_proxcot(&vs[i as usize]) {
                    setup.remove_css_class("warning");
                }
            }
        }
    });
    af
}

fn add_af_row(conn: &Rc<rusqlite::Connection>, br: &adw::ButtonRow, vs: &Rc<RefCell<Vec<Cotizante>>>, store: &gtk::gio::ListStore, pg: &adw::PreferencesGroup, lab_spinm: &gtk::Label, over: &adw::ToastOverlay, conc: &adw::EntryRow, total: &gtk::Label, caja: &adw::ComboRow, window: &adw::ApplicationWindow, vdel: &Rc<RefCell<Vec<CotizanteDel>>>, pp: &adw::PreferencesPage, sl_ca: &gtk::StringList, store_cajas: &gtk::gio::ListStore) {
    let ad = adw::AlertDialog::new(Some("Añadir cotizante"), Some(""));
    ad.add_responses(&[("canc", "Cancelar"), ("acept", "Aceptar")]);
    ad.set_default_response(Some("acept"));
    ad.set_response_appearance("acept", adw::ResponseAppearance::Suggested);
    let (slm, exp) = slm_exp_afiliados(&store);
    let dd = gtk::DropDown::builder()
        .model(&slm)
        .expression(&exp)
        .enable_search(true)
        .search_match_mode(gtk::StringFilterMatchMode::Substring)
        .build();
    ad.set_extra_child(Some(&dd));
    ad.present(Some(window));
    let br_clone = br.clone();
    let over_clone = over.clone();
    let store_clone = store.clone();
    let lab_spinm_clone = lab_spinm.clone();
    let pg_clone = pg.clone();
    let conc_clone = conc.clone();
    let total_clone = total.clone();
    let caja_clone = caja.clone();
    let window_clone = window.clone();
    let vs_clone = Rc::clone(&vs);
    let vdel_clone = Rc::clone(&vdel);
    let conn_clone = conn.clone();
    let pp_clone = pp.clone();
    let sl_ca_clone = sl_ca.clone();
    let store_cajas_clone = store_cajas.clone();
    ad.connect_response(Some("acept"), move |_, _| {
        if dd.selected_item() == None {
            return;
        }
        let s_item = dd.selected_item().unwrap().downcast::<gtk::StringObject>().unwrap().string();
        let rows = br_clone.parent().unwrap().observe_children();
        let n = rows.n_items();
        for i in 0..n - 1 {
            if rows.item(i).and_downcast::<adw::ActionRow>().unwrap().subtitle().unwrap() == s_item {
                let toast = adw::Toast::new("Ya está en la lista.");
                toast.set_timeout(2);
                over_clone.add_toast(toast);
                return;
            }
        }
        let sel = dd.selected_item().unwrap().downcast::<gtk::StringObject>().unwrap().string();
        let sel_cc = sel.split_whitespace().next().unwrap();
        let pos_store = utils::pos_cc(&store_clone, &sel_cc);
        let sli = store_clone.item(pos_store.unwrap()).and_downcast::<gtk::StringList>().unwrap();
        let setup_warning = if sli.string(3).unwrap() == utils::COTIZA[4] || sli.string(4).unwrap() == "" || sli.string(18).unwrap() == "" {
            true
        } else {
            false
        };
        let mut vs = vs_clone.borrow_mut();
        let ms = if vs.len() != 0 {vs[0].ms} else { 1 };
        let cuota = match utils::euros_to_dec(&sli.string(4).unwrap()) {
            Ok(num) => num,
            Err(_)  => Decimal::new(000, 2),
        };
        let mut del_cot = String::new();
        let mut vdel = vdel_clone.borrow_mut();
        for i in 0..vdel.len() { 
            if sli.string(0).unwrap() == vdel[i].sl.string(0).unwrap() {
                del_cot = vdel[i].cot.clone();
                vdel.remove(i);
                break;
            }
        }
        vs.push(Cotizante{sl: sli.clone(), ms, mcot: sli.string(3).unwrap().to_string(), cuota, proxcot: utils::suma_mes(&sli.string(18).unwrap()), limit_inf: utils::mes_texto_a_cod(&sli.string(18).unwrap()), limit_sup: "".to_string(), del_cot});
        total_clone.set_label(&calcula_total(&vs));
        let add_af = crear_af_row(&conn_clone, &vs_clone, &s_item, setup_warning, &lab_spinm_clone, ms, &pg_clone, &conc_clone, &total_clone, &window_clone, &vdel_clone, &pp_clone, &sl_ca_clone, &store_cajas_clone);
        let lb = br_clone.parent().and_downcast::<gtk::ListBox>().unwrap();
        lb.insert(&add_af, n as i32 - 1);
        if n == 1 {
            pg_clone.set_title("Cotizante");
            let ms_str = if vs[0].ms == 1 { "mes" } else { "meses" };
            conc_clone.set_text(&format!("{} cotiza {} {}", vs[0].sl.string(0).unwrap(), vs[0].ms, ms_str));
            sel_caja(&conn_clone, &vs, &caja_clone); 
        } else if n == 2 {
            pg_clone.set_title("Cotizantes");
            pg_clone.header_suffix().unwrap().set_visible(true);
            lab_spinm_clone.set_visible(true);
            conc_clone.set_text("");
        }
    });
}

fn calcula_total(vs: &Vec<Cotizante>) -> String {
    let mut cuotas = 0;
    let mut dinero = Decimal::new(000, 2);
    for i in 0..vs.len() {
        cuotas += vs[i].ms;
        dinero += vs[i].cuota * Decimal::from(vs[i].ms);
    }
    let cotizantes = if vs.len() == 1 { "1 cotizante" } else { &format!("{} cotizantes", vs.len()) };
    let cuotas = if cuotas == 1 { "1 cuota" } else { &format!("{} cuotas", cuotas) };
    let dinero = utils::format_euros(dinero);
    format!("{}\t{}\t{}", cotizantes, cuotas, dinero)
}

fn sel_caja(conn: &Rc<rusqlite::Connection>, vs: &Vec<Cotizante>, caja: &adw::ComboRow) {
    for j in 0..vs.len() {
        let r = conn.query_row("SELECT caja FROM cotizaciones WHERE cc=? ORDER BY mes DESC", [vs[j].sl.string(0).unwrap().to_string()], |row| {
            let mut res = false;
            let caja_ant = row.get(0).unwrap_or_else(|_| "".to_string());
            let sl_cajas = caja.model();
            if caja_ant != "" {
                let i = sl_cajas.and_downcast::<gtk::StringList>().unwrap().find(&caja_ant);
                if i != u32::MAX {
                    caja.set_selected(i);
                    res = true;
                }
            }
            Ok(res)
        });
        if r == Ok(true) { break; }
    } 
}

fn check_proxcot(vsi: &Cotizante) -> bool {
    if vsi.proxcot == "" {
        return false
    }
    let prox = utils::mes_texto_a_cod(&vsi.proxcot);
    if prox.cmp(&vsi.limit_inf) != std::cmp::Ordering::Greater {
        return false
    }
    if vsi.limit_sup != "" {
        let y: i32 = prox[0..4].parse().unwrap();
        let m: i32 = prox[4..6].parse().unwrap();
        let ma = m + vsi.ms - 1;
        let mf = ma%12;
        let yf = y + ma/12;
        let proxf = format!("{}{:02}", yf, mf);
        if proxf.cmp(&vsi.limit_sup) != std::cmp::Ordering::Less {
            return false
        }
    }
    true
}

fn slm_exp_afiliados(store: &gtk::gio::ListStore) -> (gtk::SortListModel, gtk::PropertyExpression) {
    let sl = gtk::StringList::new(&[]);
    for i in 0..store.n_items() {
        let mut cc = store.item(i).and_downcast::<gtk::StringList>().unwrap().string(0).unwrap().to_string();
        let ape = store.item(i).and_downcast::<gtk::StringList>().unwrap().string(1).unwrap();
        let nom = store.item(i).and_downcast::<gtk::StringList>().unwrap().string(2).unwrap();
        if ape != "" && nom != "" {
            cc.push_str(&format!(" | {}, {}", ape, nom));
        } else if ape != "" || nom != ""  {
            cc.push_str(&format!(" | {}{}", ape, nom));
        }
        sl.append(&cc);
    }
    let exp = gtk::PropertyExpression::new(gtk::StringObject::static_type(), gtk::Expression::NONE, "string");
    let sorter = gtk::StringSorter::new(Some(&exp));
    let slm = gtk::SortListModel::new(Some(sl), Some(sorter));
    (slm, exp)
}

fn crea_cotizaciones_afiliad(conn: &Rc<rusqlite::Connection>, combo: &gtk::DropDown, year: &gtk::SpinButton, storev: &gtk::gio::ListStore, cuotas: &gtk::Label) {
    if combo.selected_item() == None {
        return;
    }
    storev.remove_all();
    let sel = combo.selected_item().unwrap().downcast::<gtk::StringObject>().unwrap().string();
    let cc = sel.split_whitespace().next().unwrap();
    let y = year.value().to_string();
    let mut stmt = conn.prepare(&format!("SELECT num, caja, mes FROM cotizaciones WHERE cc='{}' AND mes LIKE'{}%' ORDER BY mes DESC", &cc, &y)).unwrap();
    let mut rows = stmt.query([]).unwrap();
    let mut i = 0;
    while let Some(row) = rows.next().unwrap() {
        i += 1;
        let num: i32 = row.get(0).unwrap();
        let caja: String = row.get(1).unwrap();
        let anyo_mes: String = row.get(2).unwrap();
        let mes = utils::MESES[anyo_mes[4..6].parse::<usize>().unwrap()];
        let _ = conn.query_row(&format!("SELECT fecha FROM '{}' WHERE num=? ", caja), [num], |row| {
            let fecha: String = row.get(0).unwrap();
            let sl = gtk::StringList::new(&[&format!("Movimiento {} de {}", num, caja), &mes, &utils::convert_fech(&fecha)]);
            storev.append(&sl);
            Ok(())
        });
    }
    let c = if i == 1 { "cuota" } else { "cuotas" };
    cuotas.set_label(&format!("{} {}", i, c));
}

fn crea_cotizaciones_mes(conn: &Rc<rusqlite::Connection>, combo: &gtk::DropDown, year: &gtk::SpinButton, storev: &gtk::gio::ListStore, cuotas: &gtk::Label) {
    storev.remove_all();
    let m = format!("{:02}", combo.selected()+1);
    let y = year.value().to_string();
    let mut stmt = conn.prepare(&format!("SELECT cotizaciones.cc, afiliacion.apellidos, afiliacion.nombre, cotizaciones.num, cotizaciones.caja FROM cotizaciones LEFT JOIN afiliacion ON afiliacion.cc=cotizaciones.cc WHERE cotizaciones.mes='{}{}'", &y, &m)).unwrap();
    let mut rows = stmt.query([]).unwrap();
    let mut i = 0;
    while let Some(row) = rows.next().unwrap() {
        i += 1;
        let mut cc: String = row.get(0).unwrap();
        let ape: String = row.get(1).unwrap_or_else(|_| { "".to_string() });
        let nom: String = row.get(2).unwrap_or_else(|_| { "".to_string() });
        if ape != "" && nom != "" {
            cc.push_str(&format!(" | {}, {}", ape, nom));
        } else if ape != "" || nom != ""  {
            cc.push_str(&format!(" | {}{}", ape, nom));
        }
        let num: i32 = row.get(3).unwrap();
        let caja: String = row.get(4).unwrap();
        let _ = conn.query_row(&format!("SELECT fecha FROM '{}' WHERE num=? ", caja), [num], |row| {
            let fecha: String = row.get(0).unwrap();
            let sl = gtk::StringList::new(&[&format!("Movimiento {} de {}", num, caja), &cc, &utils::convert_fech(&fecha)]);
            storev.append(&sl);
            Ok(())
        });
    }
    let c = if i == 1 { "cuota" } else { "cuotas" };
    cuotas.set_label(&format!("{} {}", i, c));
}

fn crea_cotizaciones_fechas(conn: &Rc<rusqlite::Connection>, desde: &gtk::Entry, hasta: &gtk::Entry, storev: &gtk::gio::ListStore, cuotas: &gtk::Label) {
    storev.remove_all();
    if !utils::check_date(&desde.text()) || !utils::check_date(&hasta.text()) || desde.text() == "" || hasta.text() == "" {
        cuotas.set_label("");
        return;
    }
    let desde = utils::convert_fech(&desde.text());
    let hasta = utils::convert_fech(&hasta.text());
    let sl_cajas = db_utils::lista_cajas(&conn);
    if sl_cajas.n_items() == 0 {
        return;
    }
    let mut query = String::from("SELECT ");
    for i in 0..sl_cajas.n_items() {
        query.push_str(&format!("num, '{0}' AS caja, fecha FROM '{}' WHERE etiquetas='Cotización' AND fecha BETWEEN '{}' AND '{}'", sl_cajas.string(i).unwrap(), desde, hasta));
        if i != sl_cajas.n_items()-1 {
            query.push_str(" UNION ALL SELECT ");
        }
    }
    query.push_str(" ORDER BY fecha");
    let mut stmt = conn.prepare(&query).unwrap();
    let mut rows = stmt.query([]).unwrap();
    let mut i = 0;
    while let Some(row) = rows.next().unwrap() {
        let num: i32 = row.get(0).unwrap();
        let caja: String = row.get(1).unwrap();
        let fecha: String = row.get(2).unwrap();
        let mut stmt2 = conn.prepare(&format!("SELECT cotizaciones.cc, afiliacion.apellidos, afiliacion.nombre, cotizaciones.mes FROM cotizaciones LEFT JOIN afiliacion ON afiliacion.cc=cotizaciones.cc WHERE num={} AND caja='{}'", num, caja)).unwrap();
        let mut rows2 = stmt2.query([]).unwrap();
        while let Some(row2) = rows2.next().unwrap() {
            i += 1;
            let mut cc: String = row2.get(0).unwrap();
            let ape: String = row2.get(1).unwrap_or_else(|_| { "".to_string() });
            let nom: String = row2.get(2).unwrap_or_else(|_| { "".to_string() });
            if ape != "" && nom != "" {
                cc.push_str(&format!(" | {}, {}", ape, nom));
            } else if ape != "" || nom != ""  {
                cc.push_str(&format!(" | {}{}", ape, nom));
            }
            let anyo_mes: String = row2.get(3).unwrap();
            let mes = utils::mes_cod_a_texto(&anyo_mes);
            let sl = gtk::StringList::new(&[&format!("Movimiento {} de {}", num, caja), &format!("{}: {}", cc, mes), &utils::convert_fech(&fecha)]);
            storev.append(&sl);
        }
    }
    let c = if i == 1 { "cuota" } else { "cuotas" };
    cuotas.set_label(&format!("{} {}", i, c));
}

fn crea_cotizaciones_estadillos(conn: &Rc<rusqlite::Connection>, year: &str, fb1: &gtk::FlowBox, fb2: &gtk::FlowBox) {
    fb1.remove_all();
    let mut arr = [0; 12];
    let mut stmt = conn.prepare(&format!("SELECT mes, COUNT(*) FROM cotizaciones WHERE mes LIKE '{}%' GROUP BY mes", year)).unwrap();
    let mut rows = stmt.query([]).unwrap();
    while let Some(row) = rows.next().unwrap() {
        let codmes: String = row.get(0).unwrap();
        let cant: u32 = row.get(1).unwrap();
        let i = codmes[4..].parse::<usize>().unwrap() - 1;
        arr[i] = cant;
    }
    for i in 0..arr.len() {
        let hbox = gtk::Box::new(gtk::Orientation::Horizontal, 10);
        let lmes = gtk::Label::new(Some(&(utils::MESES[i+1].to_owned()+":")));
        lmes.set_halign(gtk::Align::Start);
        lmes.set_hexpand(true);
        hbox.append(&lmes);
        let lcant = gtk::Label::new(Some(&arr[i].to_string()));
        hbox.append(&lcant);
        fb1.append(&hbox);
    }
    fb2.remove_all();
    let mut arr = [0; 12];
    let sl_cajas = db_utils::lista_cajas(&conn);
    if sl_cajas.n_items() != 0 {
        let mut query = String::from("SELECT ");
        for i in 0..sl_cajas.n_items() {
            query.push_str(&format!("num, '{0}' AS caja, strftime('%m', fecha) FROM '{}' WHERE etiquetas='Cotización' AND fecha LIKE '{}%'", sl_cajas.string(i).unwrap(), year));
            if i != sl_cajas.n_items()-1 {
                query.push_str(" UNION ALL SELECT ");
            }
        }
        let mut stmt = conn.prepare(&query).unwrap();
        let mut rows = stmt.query([]).unwrap();
        while let Some(row) = rows.next().unwrap() {
            let num: u32 = row.get(0).unwrap();
            let caja: String = row.get(1).unwrap();
            let mes: String = row.get(2).unwrap();
            let i = mes.parse::<usize>().unwrap() - 1;
            let _ = conn.query_row("SELECT COUNT(*) FROM cotizaciones WHERE num=? AND caja=?", [num.to_string(), caja], |row| {
                let cant: u32 = row.get(0).unwrap();
                arr[i] += cant;
                Ok(())
            });
        }
    }
    for i in 0..arr.len() {
        let hbox = gtk::Box::new(gtk::Orientation::Horizontal, 10);
        let lmes = gtk::Label::new(Some(&(utils::MESES[i+1].to_owned()+":")));
        lmes.set_halign(gtk::Align::Start);
        lmes.set_hexpand(true);
        hbox.append(&lmes);
        let lcant = gtk::Label::new(Some(&arr[i].to_string()));
        hbox.append(&lcant);
        fb2.append(&hbox);
    }
}
