use adw::prelude::*;
use gtk::glib;
use std::rc::Rc;
use std::cell::RefCell;
use rust_decimal::prelude::*;
use crate::utils;
use crate::db_utils;
use crate::artilugios_ca;
use crate::artilugios_af;

pub fn mov(conn: &Rc<rusqlite::Connection>, nav: &adw::NavigationView, window: &adw::ApplicationWindow, store: &gtk::gio::ListStore, wbf: &adw::WrapBox, custfilter: &gtk::CustomFilter, osl: Option<&gtk::StringList>, ocv: Option<&gtk::ColumnView>) {
    let sl = match osl {
        Some(x) => x,
        None => &gtk::StringList::new(&["","","","","","",""]),
    };
    let cv = match ocv {
        Some(x) => x,
        None => &gtk::ColumnView::new(Some(gtk::NoSelection::new(Some(gtk::StringList::new(&[]))))),
    };
    let num_upd = sl.string(0).unwrap();
    
    let vbox = gtk::Box::new(gtk::Orientation::Vertical, 0);
    let hb = adw::HeaderBar::new();
    vbox.append(&hb);
    let pp = adw::PreferencesPage::new();
    
    let pgt = adw::PreferencesGroup::new();
    let hbox = gtk::Box::new(gtk::Orientation::Horizontal, 0); //builder
    hbox.add_css_class("linked");
    hbox.set_halign(gtk::Align::Center);
    hbox.set_homogeneous(true);
    let entrada = gtk::ToggleButton::with_label("Entrada");
    entrada.connect_toggled(move |entrada| {
        if entrada.is_active() {
            entrada.add_css_class("success");
        } else {
            entrada.remove_css_class("success");
        }
    });
    hbox.append(&entrada);
    let salida = gtk::ToggleButton::with_label("Salida");
    salida.connect_toggled(move |salida| {
        if salida.is_active() {
            salida.add_css_class("error");
        } else {
            salida.remove_css_class("error");
        }
    });
    salida.set_group(Some(&entrada));
    if osl.is_some() {
        if sl.string(5).unwrap().contains('-') {
            salida.set_active(true);
        } else {
            entrada.set_active(true);
        }
    }
    hbox.append(&salida);
    pgt.add(&hbox);
    pp.add(&pgt);
    
    let pg = adw::PreferencesGroup::new();
    let imp = adw::SpinRow::with_range(0.0, 10000000.0, 1.0);
    imp.set_title("Importe (€)");
    imp.set_digits(2);
    if osl.is_some() {
        let imp_ant = sl.string(5).unwrap().split(" €").next().unwrap().replace("," ,".").replace("-", "").parse().unwrap();
        imp.set_value(imp_ant);
    }
    pg.add(&imp);
    let sl_cajas = if osl == None {
        db_utils::lista_cajas(&conn)
    } else {
        let s = store.item(0).and_downcast::<gtk::StringList>().unwrap().string(0).unwrap();
        gtk::StringList::new(&[&s])
    };
    let caja = adw::ComboRow::builder()
        .title("Caja")
        .model(&sl_cajas)
        .build();
    let i = sl_cajas.find(&store.item(0).and_downcast::<gtk::StringList>().unwrap().string(0).unwrap());
    caja.set_selected(i);
    pg.add(&caja);
    let fact = adw::EntryRow::builder()
        .title("Factura")
        .text(sl.string(2).unwrap())
        .activates_default(true)
        .build();
    pg.add(&fact);
    let conc = adw::EntryRow::builder()
        .title("Concepto")
        .text(sl.string(3).unwrap())
        .activates_default(true)
        .build();
    pg.add(&conc);
    let init = if osl == None { 
        let now = glib::DateTime::now_local().unwrap();
        format!("{:02}/{:02}/{}" ,now.day_of_month(), now.month(), now.year())
    } else {
        sl.string(1).unwrap().to_string()
    };
    let fech = adw::EntryRow::builder()
        .title("Fecha")
        .tooltip_text("dd/mm/aaaa")
        .activates_default(true)
        .text(init)
        .build();
    utils::menusuf_calendar(&fech, &utils::calendario(&fech, true));
    pg.add(&fech);
    pp.add(&pg);
    let pge = adw::PreferencesGroup::new();
    pge.set_title("Etiquetas");
    let gest = gtk::Button::from_icon_name("document-edit-symbolic");
    gest.add_css_class("flat");
    pge.set_header_suffix(Some(&gest));
    let wbm = adw::WrapBox::new();
    wbm.set_child_spacing(5);
    wbm.set_line_spacing(5);
    let ets = sl.string(4).unwrap();
    let v: Vec<&str> = ets.split(" | ").collect();
    let chs = wbf.observe_children();
    for i in 1..chs.n_items() {    
        let tglab = chs.item(i).and_downcast::<gtk::ToggleButton>().unwrap().label().unwrap();
        let tg = gtk::ToggleButton::with_label(&tglab);
        if v.contains(&tglab.as_str()) {
            tg.set_active(true);
        }
        wbm.append(&tg);
    }
    pge.add(&wbm);
    let conn_clone = conn.clone();
    let wbf_clone = wbf.clone();
    let custfilter_clone = custfilter.clone();
    let store_clone = store.clone();
    let wbm_clone = wbm.clone();
    gest.connect_clicked(move |_| {
        artilugios_ca::etiquetas(&conn_clone, &wbf_clone, &custfilter_clone , &store_clone, Some(&wbm_clone));
    });
    pp.add(&pge);
    let pga = adw::PreferencesGroup::new();
    let acept= gtk::Button::builder()
        .halign(gtk::Align::Center)
        .css_classes(["pill", "suggested-action"])
        .label("Aplicar")
        .build();
    pga.add(&acept);
    pp.add(&pga);
    vbox.append(&pp);
    let over = adw::ToastOverlay::new();
    over.set_child(Some(&vbox));
    let tit = if osl == None { "Movimiento" } else { "Editar movimiento" };
    let page = adw::NavigationPage::new(&over, tit);
    nav.push(&page);
    window.set_default_widget(Some(&acept));
    let conn_clone = conn.clone();
    let nav_clone = nav.clone();
    let store_clone = store.clone();
    let sl_clone = sl.clone();
    let cv_clone = cv.clone();
    acept.connect_clicked(move |_| {
        let toast = adw::Toast::new("");
        if !entrada.is_active() && !salida.is_active() {
            toast.set_title("Debes seleccionar entrada o salida.");
        } else if conc.text().trim() == "" {
            toast.set_title("Debe existir un concepto.");
            conc.grab_focus();
        } else if !utils::check_date(&fech.text()) || fech.text() == "" {
            toast.set_title("Debe existir una fecha válida.");
            fech.grab_focus();
        } else {
            nav_clone.pop();
            let fecha_bd = utils::convert_fech(&fech.text().to_string());
            let caja_str = caja.selected_item().unwrap().downcast::<gtk::StringObject>().unwrap().string();
            let mut ets = String::new();
            let mut first = true;
            let chs = wbm.observe_children();
            for i in 0..chs.n_items() {  
                let tg = chs.item(i).and_downcast::<gtk::ToggleButton>().unwrap();
                if tg.is_active() {
                    if first {
                        first = false;
                    } else {
                        ets.push_str(" | ");
                    }
                    ets.push_str(&tg.label().unwrap());
                }
            }
            let importe = if salida.is_active() && imp.value() > 0.0 { -imp.value() } else { imp.value() };
            let importe_s = format!("{:.2} €", importe).replace(".", ",");
            if tit == "Movimiento" {
                let query = format!("INSERT INTO '{}' (fecha, factura, concepto, importe, etiquetas) VALUES (?1, ?2, ?3, ?4, ?5)", caja_str);
                conn_clone.execute(&query, [&fecha_bd, &fact.text().to_string(), &conc.text().to_string(), &importe.to_string(), &ets]).unwrap();
                let num = conn_clone.last_insert_rowid(); //Verificar que esto no da problemas
                if caja_str == store_clone.item(0).and_downcast::<gtk::StringList>().unwrap().string(0).unwrap() {
                    let sl_mov = gtk::StringList::new(&[&num.to_string(), &fech.text(), &fact.text(), &conc.text(), &ets, &importe_s, ""]);
                    utils::recrea_lista(&store_clone, &sl_mov);
                    utils::select_key_cv(&cv_clone, &num.to_string());
                }
            } else {
                let query = format!("UPDATE '{}' SET fecha=?1, factura=?2, concepto=?3, importe=?4, etiquetas=?5 WHERE num=?6", caja_str);
                conn_clone.execute(&query, [&fecha_bd, &fact.text().to_string(), &conc.text().to_string(), &importe.to_string(), &ets, &num_upd.to_string()]).unwrap();
                let sln = gtk::StringList::new(&[&num_upd.to_string(), &fech.text(), &fact.text(), &conc.text(), &ets, &importe_s, ""]);
                utils::recrea_lista_upd(&store_clone, &sl_clone, &sln);
            }
            return;
        }
        toast.set_timeout(2);
        over.add_toast(toast);
    });
}

pub fn eliminar_mov(conn: &Rc<rusqlite::Connection>, window: &adw::ApplicationWindow, store: &gtk::gio::ListStore, sl: &gtk::StringList) {
    let num = sl.string(0).unwrap();
    let ad = adw::AlertDialog::new(Some("Elminar movimiento"), Some(&num));
    ad.add_responses(&[("canc", "Cancelar"), ("acept", "Aceptar")]);
    ad.set_default_response(Some("acept"));
    ad.set_response_appearance("acept", adw::ResponseAppearance::Destructive);
    ad.present(Some(window));
    let store_clone = store.clone();
    let sl_clone = sl.clone();
    let conn_clone = conn.clone();
    ad.connect_response(Some("acept"), move |_, _| {
        let caja_str = store_clone.item(0).and_downcast::<gtk::StringList>().unwrap().string(0).unwrap();
        let query = format!("DELETE FROM '{}' WHERE num=?", caja_str);
        conn_clone.execute(&query, [num.to_string()]).unwrap();
        utils::recrea_lista_del(&store_clone, &sl_clone);
    });
}

pub fn eliminar_cot(conn: &Rc<rusqlite::Connection>, window: &adw::ApplicationWindow, store: &gtk::gio::ListStore, sl: &gtk::StringList, store_af: &gtk::gio::ListStore) {
    let num = sl.string(0).unwrap();
    let ad = adw::AlertDialog::new(Some("Elminar movimiento"), Some(&num));
    ad.add_responses(&[("canc", "Cancelar"), ("acept", "Aceptar")]);
    ad.set_default_response(Some("acept"));
    ad.set_response_appearance("acept", adw::ResponseAppearance::Destructive);
    ad.present(Some(window));
    let store_clone = store.clone();
    let sl_clone = sl.clone();
    let conn_clone = conn.clone();
    let store_af_clone = store_af.clone();
    ad.connect_response(Some("acept"), move |_, _| {
        let caja_str = store_clone.item(0).and_downcast::<gtk::StringList>().unwrap().string(0).unwrap();
        let query = format!("DELETE FROM '{}' WHERE num=?", caja_str);
        conn_clone.execute(&query, [num.to_string()]).unwrap();
        utils::recrea_lista_del(&store_clone, &sl_clone);
        let mut vcc = Vec::new();
        let mut stmt = conn_clone.prepare("SELECT cc FROM cotizaciones WHERE caja=?1 AND num=?2").unwrap();
        let mut rows = stmt.query([caja_str.to_string(), num.to_string()]).unwrap();
        while let Some(row) = rows.next().unwrap() {
            let cc: String = row.get(0).unwrap();
            let pos = utils::pos_cc(&store_af_clone, &cc);
            if pos == None { continue; }
            vcc.push(pos.unwrap());
        }
        conn_clone.execute("DELETE FROM cotizaciones WHERE caja=? AND num=?", [caja_str.to_string(), num.to_string()]).unwrap();
        for pos in vcc {
            let sl = store_af_clone.item(pos).and_downcast::<gtk::StringList>().unwrap();
            let cc = sl.string(0).unwrap();
            if conn_clone.query_row("SELECT mes FROM cotizaciones WHERE cc=? ORDER BY mes DESC", [&cc.to_string()], |row_mes| {
                let cot = row_mes.get(0).unwrap_or_else(|_| "".to_string());
                let cot_txt = if cot != "" { utils::mes_cod_a_texto(&cot) } else { cot };
                sl.splice(18, 1, &[&cot_txt]);
                Ok(())
            }).is_err() { 
                sl.splice(18, 1, &[""]);
            }
            store_af_clone.remove(pos);
            store_af_clone.insert(pos, &sl);
        }
    });
}

pub fn transf(conn: &Rc<rusqlite::Connection>, nav: &adw::NavigationView, window: &adw::ApplicationWindow, store: &gtk::gio::ListStore, cv: &gtk::ColumnView) {    
    let vbox = gtk::Box::new(gtk::Orientation::Vertical, 0);
    let hb = adw::HeaderBar::new();
    vbox.append(&hb);
    let pp = adw::PreferencesPage::new();    
    let pg = adw::PreferencesGroup::new();
    let imp = adw::SpinRow::with_range(0.0, 10000000.0, 1.0);
    imp.set_title("Importe (€)");
    imp.set_digits(2);
    pg.add(&imp);
    let sl_cajas = db_utils::lista_cajas(&conn);
    let sel = sl_cajas.find(&store.item(0).and_downcast::<gtk::StringList>().unwrap().string(0).unwrap());
    let caja_s = adw::ComboRow::builder()
        .title("Caja salida")
        .model(&sl_cajas)
        .selected(sel)
        .build();
    pg.add(&caja_s);
    let sel2 = if sel == 0 { 1 } else { 0 }; 
    let caja_e = adw::ComboRow::builder()
        .title("Caja entrada")
        .model(&sl_cajas)
        .selected(sel2)
        .build();
    pg.add(&caja_e);
    let init = format!("Transferencia de {} a {}", caja_s.selected_item().unwrap().downcast::<gtk::StringObject>().unwrap().string(), caja_e.selected_item().unwrap().downcast::<gtk::StringObject>().unwrap().string());
    let conc = adw::EntryRow::builder()
        .title("Concepto")
        .text(&init)
        .activates_default(true)
        .build();
    pg.add(&conc);
    let now = glib::DateTime::now_local().unwrap();
    let today = format!("{:02}/{:02}/{}" ,now.day_of_month(), now.month(), now.year());
    let fech = adw::EntryRow::builder()
        .title("Fecha")
        .tooltip_text("dd/mm/aaaa")
        .activates_default(true)
        .text(today)
        .build();
    utils::menusuf_calendar(&fech, &utils::calendario(&fech, true));
    pg.add(&fech);
    pp.add(&pg);
    let pga = adw::PreferencesGroup::new();
    let acept= gtk::Button::builder()
        .halign(gtk::Align::Center)
        .css_classes(["pill", "suggested-action"])
        .label("Aplicar")
        .build();
    pga.add(&acept);
    pp.add(&pga);
    vbox.append(&pp);
    let over = adw::ToastOverlay::new();
    over.set_child(Some(&vbox));
    let page = adw::NavigationPage::new(&over, "Transferencia");
    nav.push(&page);
    window.set_default_widget(Some(&acept));
    
    let caja_e_clone = caja_e.clone();
    let conc_clone = conc.clone();
    caja_s.connect_selected_notify(move |caja_s| {
        let txt = format!("Transferencia de {} a {}", caja_s.selected_item().unwrap().downcast::<gtk::StringObject>().unwrap().string(), caja_e_clone.selected_item().unwrap().downcast::<gtk::StringObject>().unwrap().string());
        conc_clone.set_text(&txt);
    });
    let caja_s_clone = caja_s.clone();
    let conc_clone = conc.clone();
    caja_e.connect_selected_notify(move |caja_e| {
        let txt = format!("Transferencia de {} a {}", caja_s_clone.selected_item().unwrap().downcast::<gtk::StringObject>().unwrap().string(), caja_e.selected_item().unwrap().downcast::<gtk::StringObject>().unwrap().string());
        conc_clone.set_text(&txt);
    });
    
    
    let conn_clone = conn.clone();
    let nav_clone = nav.clone();
    let store_clone = store.clone();
    let cv_clone = cv.clone();
    acept.connect_clicked(move |_| {
        let toast = adw::Toast::new("");
        if imp.value() == 0.0 {
            toast.set_title("El importe no puede ser 0,00 €.");
            imp.grab_focus();
        } else if caja_s.selected() == caja_e.selected() {
            toast.set_title("No se puede transferir a la misma caja.");
        } else if conc.text().trim() == "" {
            toast.set_title("Debe existir un concepto.");
            conc.grab_focus();
        } else if !utils::check_date(&fech.text()) || fech.text() == "" {
            toast.set_title("Debe existir una fecha válida.");
            fech.grab_focus();          
        } else {
            nav_clone.pop();
            
            let fecha_bd = utils::convert_fech(&fech.text().to_string());
            let importe = imp.value();
            let caja_s_str = caja_s.selected_item().unwrap().downcast::<gtk::StringObject>().unwrap().string();
            let caja_e_str = caja_e.selected_item().unwrap().downcast::<gtk::StringObject>().unwrap().string();
            let mut num = 0;
            let query = format!("INSERT INTO '{}' (fecha, concepto, importe) VALUES (?1, ?2, ?3)", caja_s_str);
            conn_clone.execute(&query, [&fecha_bd, &conc.text().to_string(), &(-importe).to_string()]).unwrap();
            if sel == caja_s.selected() {
                num = conn_clone.last_insert_rowid();
            }
            let query = format!("INSERT INTO '{}' (fecha, concepto, importe) VALUES (?1, ?2, ?3)", caja_e_str);
            conn_clone.execute(&query, [&fecha_bd, &conc.text().to_string(), &importe.to_string()]).unwrap();
            if sel == caja_e.selected() {
                num = conn_clone.last_insert_rowid();
            }
            if num > 0 {
                let importe = if sel == caja_s.selected() {
                    format!("-{:.2} €", importe).replace(".", ",")
                } else {
                    format!("{:.2} €", importe).replace(".", ",")
                };
                let sl_mov = gtk::StringList::new(&[&num.to_string(), &fech.text(), "", &conc.text(), "", &importe, ""]);
                utils::recrea_lista(&store_clone, &sl_mov);
                utils::select_key_cv(&cv_clone, &num.to_string());
            }
            return;
        }
        toast.set_timeout(2);
        over.add_toast(toast);
    });
}

pub fn crear_caja(conn: &Rc<rusqlite::Connection>, nav: &adw::NavigationView, ostack: Option<&adw::ViewStack>, window: &adw::ApplicationWindow, osearchentry: Option<&gtk::SearchEntry>, odd: Option<&gtk::DropDown>, ostore: Option<&gtk::gio::ListStore>, ostore_af: Option<&gtk::gio::ListStore>) {
    let first = if ostack == None { false } else { true };
    let stack = match ostack {
        Some(x) => x,
        None => &adw::ViewStack::new(),
    };
    let searchentry = match osearchentry {
        Some(x) => x,
        None => &gtk::SearchEntry::new(),
    };
    let dd = match odd {
        Some(x) => x,
        None => &gtk::DropDown::from_strings(&[""]),
    };
    let store = match ostore {
        Some(x) => x,
        None => &gtk::gio::ListStore::new::<gtk::StringList>(),
    };
    let store_af = match ostore_af {
        Some(x) => x,
        None => &gtk::gio::ListStore::new::<gtk::StringList>(),
    };
    let vbox = gtk::Box::new(gtk::Orientation::Vertical, 0);
    let hb = adw::HeaderBar::new();
    vbox.append(&hb);
    let pp = adw::PreferencesPage::new();
    let pg = adw::PreferencesGroup::new();
    let nom = adw::EntryRow::builder()
        .title("Nombre")
        .activates_default(true)
        .build();
    pg.add(&nom);
    let saldo = adw::SpinRow::with_range(-10000000.0, 10000000.0, 1.0);
    saldo.set_title("Saldo incial (€)");
    saldo.set_digits(2);
    saldo.set_value(0.0);
    pg.add(&saldo);
    let now = glib::DateTime::now_local().unwrap();
    let today = format!("{:02}/{:02}/{}", now.day_of_month(), now.month(), now.year());
    let fecha = adw::EntryRow::builder()
        .title("Fecha")
        .tooltip_text("dd/mm/aaaa")
        .activates_default(true)
        .text(today)
        .build();
    utils::menusuf_calendar(&fecha, &utils::calendario(&fecha, true));
    pg.add(&fecha);
    pp.add(&pg);
    let pga = adw::PreferencesGroup::new();
    let acept= gtk::Button::builder()
        .halign(gtk::Align::Center)
        .css_classes(["pill", "suggested-action"])
        .label("Aplicar")
        .build();
    pga.add(&acept);
    pp.add(&pga);
    vbox.append(&pp);
    let over = adw::ToastOverlay::new();
    over.set_child(Some(&vbox));
    acept.grab_focus();
    let page = adw::NavigationPage::new(&over, "Crear caja");
    nav.push(&page);
    window.set_default_widget(Some(&acept));
    nom.connect_changed(move |nom| {
        if nom.text_length() != 0 {
            nom.remove_css_class("error");
        }
    });
    let conn_clone = conn.clone();
    let nav_clone = nav.clone();
    let stack_clone = stack.clone();
    let searchentry_clone = searchentry.clone();
    let window_clone = window.clone();
    let dd_clone = dd.clone();
    let store_clone = store.clone();
    let store_af_clone = store_af.clone();
    acept.connect_clicked(move |_| {
        let toast = adw::Toast::new("");
        if nom.text().trim() == "" {
            toast.set_title("Debes introducir un nombre.");
            nom.add_css_class("error");
            nom.grab_focus();
        } else if !utils::check_date(&fecha.text()) {
            toast.set_title("Error fecha.");
            fecha.grab_focus();
        } else {
            let r = conn_clone.execute(&format!(
                "CREATE TABLE '{}' (
                    num         INTEGER PRIMARY KEY,
                    fecha       DATE,
                    factura     TEXT,
                    concepto    TEXT,
                    etiquetas   TEXT,
                    importe     REAL
                )", nom.text()),
                (),
            ).unwrap_or_else(|_| {
                1000
            });
            if r == 1000 {
                toast.set_title("Nombre no válido.");
                nom.add_css_class("error");
                nom.grab_focus();
            } else {
                nav_clone.pop();
                conn_clone.execute(&format!("INSERT INTO '{}' (fecha, concepto, importe) VALUES ('{}', '{}', {})", nom.text(), utils::convert_fech(&fecha.text()), "Saldo inicial", saldo.value()), ()).unwrap();
                if first {
                    let ch = stack_clone.child_by_name("ca").and_downcast::<adw::StatusPage>().unwrap();
                    stack_clone.remove(&ch);
                    artilugios_ca::ca(&conn_clone, &stack_clone, &nav_clone, &window_clone, &searchentry_clone, &store_clone, &store_af_clone);
                } else {
                    let m = dd_clone.model();
                    let sl = m.and_downcast::<gtk::StringList>().unwrap();
                    sl.append(&nom.text());
                }
                return;
            }
        }
        toast.set_timeout(2);
        over.add_toast(toast);
    });
}

pub fn eliminar_caja(conn: &Rc<rusqlite::Connection>, window: &adw::ApplicationWindow, store_ca: &gtk::gio::ListStore, store_af: &gtk::gio::ListStore, dd: &gtk::DropDown, stack: &adw::ViewStack, nav: &adw::NavigationView, searchentry: &gtk::SearchEntry) {
    let ad = adw::AlertDialog::new(Some("Elminar caja"), Some("Esto eliminará también las cotizaciones y el saldo que contenga."));
    ad.add_responses(&[("canc", "Cancelar"), ("acept", "Aceptar")]);
    ad.set_default_response(Some("acept"));
    ad.set_response_appearance("acept", adw::ResponseAppearance::Destructive);
    let pg = adw::PreferencesGroup::new();
    let model = dd.model().unwrap();
    let sel = dd.selected();
    let caja = adw::ComboRow::builder()
        .title("Caja")
        .model(&model)
        .selected(sel)
        .build();
    pg.add(&caja);
    ad.set_extra_child(Some(&pg));
    ad.present(Some(window));
    let conn_clone = conn.clone();
    let stack_clone = stack.clone();
    let nav_clone = nav.clone();
    let window_clone = window.clone();
    let searchentry_clone = searchentry.clone();
    let store_ca_clone = store_ca.clone();
    let store_af_clone = store_af.clone();
    let dd_clone = dd.clone();
    ad.connect_response(Some("acept"), move |_, _| {
        let caja_str = caja.selected_item().unwrap().downcast::<gtk::StringObject>().unwrap().string();
        let _ = conn_clone.execute(&format!("DROP TABLE '{}'", caja_str), ());
        let r = conn_clone.execute("DELETE FROM cotizaciones WHERE caja=?", [caja_str.to_string()]).unwrap();
        if r > 0 {
            store_af_clone.remove_all();
            artilugios_af::crea_lista(&store_af_clone, &conn_clone);
        }
        if model.n_items() > 1 {
            if caja.selected() == dd_clone.selected() {
                if dd_clone.selected() == 0 {
                    dd_clone.set_selected(1);
                } else {
                    dd_clone.set_selected(0);
                }
            }
            let sl = model.downcast_ref::<gtk::StringList>().unwrap();
            sl.remove(caja.selected());
        } else {
            let ch = stack_clone.child_by_name("ca").and_downcast::<gtk::Box>().unwrap();
            stack_clone.remove(&ch);
            artilugios_ca::ca_prev(&conn_clone, &stack_clone, &nav_clone, &window_clone, &searchentry_clone, &store_ca_clone, &store_af_clone)
        }
    });
}

pub fn mod_caja(conn: &Rc<rusqlite::Connection>, window: &adw::ApplicationWindow, store: &gtk::gio::ListStore, dd: &gtk::DropDown, nav: &adw::NavigationView) {
    let vbox = gtk::Box::new(gtk::Orientation::Vertical, 0);
    let hb = adw::HeaderBar::new();
    vbox.append(&hb);
    let pp = adw::PreferencesPage::new();
    let pg = adw::PreferencesGroup::new();
    let model = dd.model().unwrap();
    let sel = dd.selected();
    let caja = adw::ComboRow::builder()
        .title("Caja")
        .model(&model)
        .selected(sel)
        .build();
    pg.add(&caja);
    let nom = adw::EntryRow::builder()
        .title("Nuevo nombre")
        .activates_default(true)
        .build();
    pg.add(&nom);
    pp.add(&pg);
    let pga = adw::PreferencesGroup::new();
    let acept= gtk::Button::builder()
        .halign(gtk::Align::Center)
        .css_classes(["pill", "suggested-action"])
        .label("Aplicar")
        .build();
    pga.add(&acept);
    pp.add(&pga);
    vbox.append(&pp);
    let over = adw::ToastOverlay::new();
    over.set_child(Some(&vbox));
    acept.grab_focus();
    let page = adw::NavigationPage::new(&over, "Renombrar caja");
    nav.push(&page);
    window.set_default_widget(Some(&acept));
    let conn_clone = conn.clone();
    let nav_clone = nav.clone();
    let store_clone = store.clone();
    acept.connect_clicked(move |_| {
        let toast = adw::Toast::new("");
        if nom.text().trim() == "" {
            toast.set_title("Debes introducir un nombre.");
            nom.add_css_class("error");
            nom.grab_focus();
        } else {
            let caja_str = caja.selected_item().unwrap().downcast::<gtk::StringObject>().unwrap().string();
            let r = conn_clone.execute(&format!("ALTER TABLE '{}' RENAME TO '{}'", caja_str ,nom.text()), ()).unwrap_or_else(|_| {
                1000
            });
            if r == 1000 {
                toast.set_title("Nombre no válido.");
                nom.add_css_class("error");
                nom.grab_focus();
            } else {
                nav_clone.pop();
                conn_clone.execute("UPDATE cotizaciones SET caja=?1 WHERE caja=?2", [nom.text().to_string(), caja_str.to_string()]).unwrap();
                let sl = model.downcast_ref::<gtk::StringList>().unwrap();
                sl.splice(caja.selected(), 1, &[&nom.text()]);
                if sel == caja.selected() {
                    let m = gtk::StringList::new(&[&nom.text()]);
                    store_clone.splice(0, 1, &[m]);
                }
                return;
            }
        }
        toast.set_timeout(2);
        over.add_toast(toast);
    });
}

pub fn devolver(conn: &Rc<rusqlite::Connection>, nav: &adw::NavigationView, window: &adw::ApplicationWindow, store: &gtk::gio::ListStore, store_af: &gtk::gio::ListStore, sel: &gtk::SingleSelection) {
    let vbox = gtk::Box::new(gtk::Orientation::Vertical, 0);
    let hb = adw::HeaderBar::new();
    vbox.append(&hb);
    let pp = adw::PreferencesPage::new();
    let pg = adw::PreferencesGroup::new();
    let num = sel.selected_item().and_downcast::<gtk::StringList>().unwrap().string(0).unwrap();
    let caja = store.item(0).and_downcast::<gtk::StringList>().unwrap().string(0).unwrap();
    let slc = gtk::StringList::new(&[]);
    let mut stmt = conn.prepare("SELECT DISTINCT cc FROM cotizaciones WHERE caja=?1 AND num=?2").unwrap();
    let mut rows = stmt.query([caja.to_string(), num.to_string()]).unwrap();
    while let Some(row) = rows.next().unwrap() {
        let mut cc: String = row.get(0).unwrap();
        let pos = utils::pos_cc(&store_af, &cc);
        if pos != None {
            let ape = store_af.item(pos.unwrap()).and_downcast::<gtk::StringList>().unwrap().string(1).unwrap();
            let nom = store_af.item(pos.unwrap()).and_downcast::<gtk::StringList>().unwrap().string(2).unwrap();
            if ape != "" && nom != "" {
                cc.push_str(&format!(" | {}, {}", ape, nom));
            } else if ape != "" || nom != ""  {
                cc.push_str(&format!(" | {}{}", ape, nom));
            }
        }
        slc.append(&cc);
    }
    if slc.n_items() == 0 {
        utils::adialog("No hay cuotas a devolver de esta cotización.", &window);
        return;
    }
    let exp = gtk::PropertyExpression::new(gtk::StringObject::static_type(), gtk::Expression::NONE, "string");
    let sorter = gtk::StringSorter::new(Some(&exp));
    let slm = gtk::SortListModel::new(Some(slc), Some(sorter));
    let cot = adw::ComboRow::builder()
        .title("Cotizante")
        .model(&slm)
        .expression(&exp)
        .enable_search(true)
        .search_match_mode(gtk::StringFilterMatchMode::Substring)
        .build();
    pg.add(&cot);
    let ccan = cot.selected_item().unwrap().downcast::<gtk::StringObject>().unwrap().string();
    let cc = ccan.split(" | ").next().unwrap();
    let text = format!("{} devuelve cotización", cc);
    let conc = adw::EntryRow::builder()
        .title("Concepto")
        .text(&text)
        .activates_default(true)
        .build();
    let conc_clone = conc.clone();
    cot.connect_selected_notify(move |cot| {
        let ccan = cot.selected_item().unwrap().downcast::<gtk::StringObject>().unwrap().string();
        let cc = ccan.split(" | ").next().unwrap();
        conc_clone.set_text(&format!("{} devuelve cotización", cc));
    });
    pg.add(&conc);
    let now = glib::DateTime::now_local().unwrap();
    let init = format!("{:02}/{:02}/{}" ,now.day_of_month(), now.month(), now.year());
    let fech = adw::EntryRow::builder()
        .title("Fecha")
        .tooltip_text("dd/mm/aaaa")
        .activates_default(true)
        .text(init)
        .build();
    utils::menusuf_calendar(&fech, &utils::calendario(&fech, true));
    pg.add(&fech);
    pp.add(&pg);
    let pga = adw::PreferencesGroup::new();
    let acept= gtk::Button::builder()
        .halign(gtk::Align::Center)
        .css_classes(["pill", "suggested-action"])
        .label("Aplicar")
        .build();
    pga.add(&acept);
    pp.add(&pga);
    vbox.append(&pp);
    let over = adw::ToastOverlay::new();
    over.set_child(Some(&vbox));
    let page = adw::NavigationPage::new(&over, "Devolver cuota");
    nav.push(&page);
    window.set_default_widget(Some(&acept));
    let conn_clone = conn.clone();
    let nav_clone = nav.clone();
    let store_clone = store.clone();
    let store_af_clone = store_af.clone();
    acept.connect_clicked(move |_| {
        let toast = adw::Toast::new("");
        if conc.text().trim() == "" {
            toast.set_title("Debe existir un concepto.");
            conc.grab_focus();
        } else if !utils::check_date(&fech.text()) || fech.text() == "" {
            toast.set_title("Debe existir una fecha válida.");
            fech.grab_focus();
        } else {
            nav_clone.pop();
            let fecha_bd = utils::convert_fech(&fech.text().to_string());
            let ccan = cot.selected_item().unwrap().downcast::<gtk::StringObject>().unwrap().string();
            let cc = ccan.split(" | ").next().unwrap();
            let tx = conn_clone.unchecked_transaction().unwrap();
            let mut meses = 0;
            let _ = tx.query_row("SELECT Count(cc) FROM cotizaciones WHERE caja=?1 AND num=?2 AND cc=?3", [&caja.to_string(), &num.to_string(), cc], |row| {
                meses = row.get(0).unwrap();
                Ok(())
            });
            let mut cuota = 0.0;
            let _ = tx.query_row("SELECT cuota FROM afiliacion WHERE cc=?", [cc], |row| {
                cuota = row.get(0).unwrap();
                Ok(())
            });
            let importe = -meses as f64 * cuota;
            let importe_s = format!("{:.2} €", importe).replace(".", ",");
            let query = format!("INSERT INTO '{}' (fecha, concepto, importe) VALUES (?1, ?2, ?3)", caja);
            tx.execute(&query, [&fecha_bd, &conc.text().to_string(), &importe.to_string()]).unwrap();
            let num_insert = conn_clone.last_insert_rowid();
            let sl_mov = gtk::StringList::new(&[&num_insert.to_string(), &fech.text(), "", &conc.text(), "", &importe_s, ""]);
            utils::recrea_lista(&store_clone, &sl_mov);
            tx.execute("DELETE FROM cotizaciones WHERE caja=?1 AND num=?2 AND cc=?3", [&caja.to_string(), &num.to_string(), cc]).unwrap();
            let pos = utils::pos_cc(&store_af_clone, &cc);
            if pos != None {
                let m = store_af_clone.item(pos.unwrap()).and_downcast::<gtk::StringList>().unwrap();
                if tx.query_row("SELECT mes FROM cotizaciones WHERE cc=? ORDER BY mes DESC", [cc], |row_mes| {
                        let cot = row_mes.get(0).unwrap_or_else(|_| "".to_string());
                        let cot_txt = if cot != "" { utils::mes_cod_a_texto(&cot) } else { cot };
                        m.splice(18, 1, &[&cot_txt]);
                        Ok(())
                    }   
                ).is_err() { 
                    m.splice(18, 1, &[""]);
                }
                store_af_clone.remove(pos.unwrap());
                store_af_clone.insert(pos.unwrap(), &m);
            }
            let _ = tx.commit();
            return;
        }
        toast.set_timeout(2);
        over.add_toast(toast);
    });
}

pub fn cuadrar(conn: &Rc<rusqlite::Connection>, nav: &adw::NavigationView, window: &adw::ApplicationWindow, store: &gtk::gio::ListStore, cv: &gtk::ColumnView) {
    let vbox = gtk::Box::new(gtk::Orientation::Vertical, 0);
    let hb = adw::HeaderBar::new();
    vbox.append(&hb);
    let pp = adw::PreferencesPage::new();
    let pg = adw::PreferencesGroup::new();
    let sl_cajas = db_utils::lista_cajas(&conn);
    let caja = adw::ComboRow::builder()
        .title("Caja")
        .model(&sl_cajas)
        .build();
    let i = sl_cajas.find(&store.item(0).and_downcast::<gtk::StringList>().unwrap().string(0).unwrap());
    caja.set_selected(i);
    pg.add(&caja);
    pp.add(&pg);
    let pg2 = adw::PreferencesGroup::new();
    let fb = gtk::FlowBox::new();
    fb.set_selection_mode(gtk::SelectionMode::None);
    fb.set_min_children_per_line(2);
    pg2.add(&fb);
    pp.add(&pg2);
    let pg3 = adw::PreferencesGroup::new();
    let hbox = gtk::Box::new(gtk::Orientation::Horizontal, 10);
    hbox.set_halign(gtk::Align::Center);
    let lab = gtk::Label::new(Some("Total (€):"));
    hbox.append(&lab);
    let total = gtk::SpinButton::with_range(0.0, 10000000.0, 1.0);
    total.set_digits(2);
    total.set_activates_default(true);
    hbox.append(&total);
    pg3.add(&hbox);
    pp.add(&pg3);
    let vals: [f64; 15] = [500.0, 200.0, 100.0, 50.0, 20.0, 10.0, 5.0, 2.0, 1.0, 0.5, 0.2, 0.1, 0.05, 0.02, 0.01];
    let rs: Rc<RefCell<[u16; 15]>> = Rc::new(RefCell::new([0,0,0,0,0,0,0,0,0,0,0,0,0,0,0]));
    for i in 0..vals.len() {
        let hbox = gtk::Box::new(gtk::Orientation::Horizontal, 10);
        let s;
        if vals[i].fract() == 0.0 { s = format!("{:.0} €:", vals[i]) }
        else { s = format!("{:.2} €:", vals[i]).replace(".", ",") }
        let lab = gtk::Label::new(Some(&s));
        lab.set_halign(gtk::Align::End);
        lab.set_hexpand(true);
        hbox.append(&lab);
        let spin = gtk::SpinButton::with_range(0.0, 65535.0, 1.0);
        hbox.append(&spin);
        fb.append(&hbox);
        let rs_clone = rs.clone();
        let total_clone = total.clone();
        spin.connect_changed(move |spin| {
            let mut rs = rs_clone.borrow_mut();
            rs[i] = spin.text().parse().unwrap_or_else(|_| { 0 });
            let mut tot = 0.0;
            for j in 0..vals.len() {
                tot += rs[j] as f64 * vals[j];
            }
            total_clone.set_value(tot);
        });
    }
    let pga = adw::PreferencesGroup::new();
    let acept= gtk::Button::builder()
        .halign(gtk::Align::Center)
        .css_classes(["pill", "suggested-action"])
        .label("Aplicar")
        .build();
    pga.add(&acept);
    pp.add(&pga);
    vbox.append(&pp);
    window.set_default_widget(Some(&acept));
    let page = adw::NavigationPage::new(&vbox, "Cuadrar caja");
    nav.push(&page);
    let conn_clone = conn.clone();
    let nav_clone = nav.clone();
    let store_clone = store.clone();
    let cv_clone = cv.clone();
    acept.connect_clicked(move |_| {
        nav_clone.pop();
        let now = glib::DateTime::now_local().unwrap();
        let fecha_bd = format!("{}-{:02}-{:02}", now.year(), now.month(), now.day_of_month());
        let caja_str = caja.selected_item().unwrap().downcast::<gtk::StringObject>().unwrap().string();
        let mut saldo = 0.0;
        let _ = conn_clone.query_row(&format!("SELECT SUM(importe) FROM '{}'", caja_str), (), |row| {
            saldo = row.get(0).unwrap();
            Ok(())
        });
        let importe = total.value() - saldo;
        let query = format!("INSERT INTO '{}' (fecha, concepto, importe) VALUES (?1, ?2, ?3)", caja_str);
        conn_clone.execute(&query, [&fecha_bd, "Cuadre de caja", &importe.to_string()]).unwrap();
        let num = conn_clone.last_insert_rowid();
        if caja_str == store_clone.item(0).and_downcast::<gtk::StringList>().unwrap().string(0).unwrap() {
            let fecha = format!("{:02}/{:02}/{}", now.day_of_month(), now.month(), now.year());
            let importe_s = format!("{:.2} €", importe).replace(".", ",");
            let sl_mov = gtk::StringList::new(&[&num.to_string(), &fecha, "", "Cuadre de caja", "", &importe_s, ""]);
            utils::recrea_lista(&store_clone, &sl_mov);
            utils::select_key_cv(&cv_clone, &num.to_string());
        }
    });
}

pub fn saldos(conn: &Rc<rusqlite::Connection>, nav: &adw::NavigationView) {
    let vbox = gtk::Box::new(gtk::Orientation::Vertical, 10);
    let hb = adw::HeaderBar::new();
    vbox.append(&hb);
    let stack = adw::ViewStack::new();
    let switcher = adw::ViewSwitcher::builder()
        .stack(&stack)
        .policy(adw::ViewSwitcherPolicy::Wide)
        .halign(gtk::Align::Center)
        .build();
    vbox.append(&switcher);
    vbox.append(&stack);
    let slc = db_utils::lista_cajas(&conn);
    
    let vbox1 = gtk::Box::new(gtk::Orientation::Vertical, 0);
    let pp1 = adw::PreferencesPage::new();
    let pg11 = adw::PreferencesGroup::new();
    let now = glib::DateTime::now_local().unwrap();
    let init = format!("{:02}/{:02}/{}", now.day_of_month(), now.month(), now.year());
    let has = adw::EntryRow::builder()
        .title("Hasta")
        .tooltip_text("dd/mm/aaaa")
        .text(&init)
        .build();
    utils::menusuf_calendar(&has, &utils::calendario(&has, true));
    pg11.add(&has);
    pp1.add(&pg11);
    let pg21 = adw::PreferencesGroup::new();
    for i in 0..slc.n_items() {
        let caja = slc.item(i).and_downcast::<gtk::StringObject>().unwrap();
        let cr = adw::ActionRow::builder()
            .title(&caja)
            .subtitle("0,00 €")
            .css_classes(["property"])
            .build();
        pg21.add(&cr);
    }
    let tr = adw::ActionRow::builder()
        .title("Total")
        .subtitle("0,00 €")
        .css_classes(["property"])
        .build();
    pg21.add(&tr);
    totales(&conn, &tr, &has.text(), &slc);
    pp1.add(&pg21);
    vbox1.append(&pp1);
    let slc_clone = slc.clone();
    let conn_clone = conn.clone();
    has.connect_changed(move |has| {
        totales(&conn_clone, &tr, &has.text(), &slc_clone);
    });
    stack.add_titled_with_icon(&vbox1, None, "Totales", "accessories-calculator-symbolic");

    let vbox2 = gtk::Box::new(gtk::Orientation::Vertical, 0);
    let pp2 = adw::PreferencesPage::new();
    let pg21 = adw::PreferencesGroup::new();
    let inity = format!("01/01/{}", now.year());
    let desd = adw::EntryRow::builder()
        .title("Desde")
        .tooltip_text("dd/mm/aaaa")
        .text(&inity)
        .build();
    utils::menusuf_calendar(&desd, &utils::calendario(&desd, true));
    pg21.add(&desd);
    let hast = adw::EntryRow::builder()
        .title("Hasta")
        .tooltip_text("dd/mm/aaaa")
        .text(&init)
        .build();
    utils::menusuf_calendar(&hast, &utils::calendario(&hast, true));
    pg21.add(&hast);
    let caja = adw::ComboRow::builder()
        .title("Caja")
        .model(&slc)
        .build();
    pg21.add(&caja);
    pp2.add(&pg21);
    let pg22 = adw::PreferencesGroup::new();
    let entradas = adw::ActionRow::builder()
        .title("Entradas")
        .css_classes(["property"])
        .build();
    pg22.add(&entradas);
    let salidas = adw::ActionRow::builder()
        .title("Salidas")
        .css_classes(["property"])
        .build();
    pg22.add(&salidas);
    let total = adw::ActionRow::builder()
        .title("Total")
        .css_classes(["property"])
        .build();
    cajas_saldos(&conn, &desd.text(), &hast.text(), &caja, &entradas, &salidas, &total);
    pg22.add(&total);
    pp2.add(&pg22);
    vbox2.append(&pp2);
    let conn_clone = conn.clone();
    let hast_clone = hast.clone();
    let caja_clone = caja.clone();
    let entradas_clone = entradas.clone();
    let salidas_clone = salidas.clone();
    let total_clone = total.clone();
    desd.connect_changed(move |desd| {
        cajas_saldos(&conn_clone, &desd.text(), &hast_clone.text(), &caja_clone, &entradas_clone, &salidas_clone, &total_clone);
    });
    let conn_clone = conn.clone();
    let desd_clone = desd.clone();
    let caja_clone = caja.clone();
    let entradas_clone = entradas.clone();
    let salidas_clone = salidas.clone();
    let total_clone = total.clone();
    hast.connect_changed(move |hast| {
        cajas_saldos(&conn_clone, &desd_clone.text(), &hast.text(), &caja_clone, &entradas_clone, &salidas_clone, &total_clone);
    });
    let conn_clone = conn.clone();
    let desd_clone = desd.clone();
    let hast_clone = hast.clone();
    caja.connect_selected_notify(move |caja| {
        cajas_saldos(&conn_clone, &desd_clone.text(), &hast_clone.text(), &caja, &entradas, &salidas, &total);
    });
    stack.add_titled_with_icon(&vbox2, None, "Cajas", "view-list-symbolic");
    
    let vbox3 = gtk::Box::new(gtk::Orientation::Vertical, 0);
    let pp3 = adw::PreferencesPage::new();
    let pg31 = adw::PreferencesGroup::new();
    let desde = adw::EntryRow::builder()
        .title("Desde")
        .tooltip_text("dd/mm/aaaa")
        .text(inity)
        .build();
    utils::menusuf_calendar(&desde, &utils::calendario(&desde, true));
    pg31.add(&desde);
    let hasta = adw::EntryRow::builder()
        .title("Hasta")
        .tooltip_text("dd/mm/aaaa")
        .text(init)
        .build();
    utils::menusuf_calendar(&hasta, &utils::calendario(&hasta, true));
    pg31.add(&hasta);
    let sle = gtk::StringList::new(&[]);
    let mut stmt = conn.prepare("SELECT nombre FROM etiquetas").unwrap();
    let mut rows = stmt.query([]).unwrap();
    while let Some(row) = rows.next().unwrap() {
        let m: String = row.get(0).unwrap();
        sle.append(&m);
    }
    let et = adw::ComboRow::builder()
        .title("Etiqueta")
        .model(&sle)
        .build();
    pg31.add(&et);
    pp3.add(&pg31);
    let pg32 = adw::PreferencesGroup::new();
    let ent = adw::ActionRow::builder()
        .title("Entradas")
        .css_classes(["property"])
        .build();
    pg32.add(&ent);
    let sal = adw::ActionRow::builder()
        .title("Salidas")
        .css_classes(["property"])
        .build();
    pg32.add(&sal);
    let tot = adw::ActionRow::builder()
        .title("Total")
        .css_classes(["property"])
        .build();
    etiquetas_saldos(&conn, &desd.text(), &hast.text(), &et, &ent, &sal, &tot, &slc);
    pg32.add(&tot);
    pp3.add(&pg32);
    vbox3.append(&pp3);
    let conn_clone = conn.clone();
    let hasta_clone = hasta.clone();
    let et_clone = et.clone();
    let ent_clone = ent.clone();
    let sal_clone = sal.clone();
    let tot_clone = tot.clone();
    let slc_clone = slc.clone();
    desde.connect_changed(move |desde| {
        etiquetas_saldos(&conn_clone, &desde.text(), &hasta_clone.text(), &et_clone, &ent_clone, &sal_clone, &tot_clone, &slc_clone);
    });
    let conn_clone = conn.clone();
    let desde_clone = desde.clone();
    let et_clone = et.clone();
    let ent_clone = ent.clone();
    let sal_clone = sal.clone();
    let tot_clone = tot.clone();
    let slc_clone = slc.clone();
    hasta.connect_changed(move |hasta| {
        etiquetas_saldos(&conn_clone, &desde_clone.text(), &hasta.text(), &et_clone, &ent_clone, &sal_clone, &tot_clone, &slc_clone);
    });
    let conn_clone = conn.clone();
    let desde_clone = desde.clone();
    let hasta_clone = hasta.clone();
    et.connect_selected_notify(move |et| {
        etiquetas_saldos(&conn_clone, &desde_clone.text(), &hasta_clone.text(), &et, &ent, &sal, &tot, &slc);
    });
    stack.add_titled_with_icon(&vbox3, None, "Etiquetas", "user-bookmarks-symbolic");
    
    let page = adw::NavigationPage::new(&vbox, "Saldos");
    nav.push(&page);
}

fn totales(conn: &Rc<rusqlite::Connection>, total: &adw::ActionRow, hasta: &str, sl: &gtk::StringList) {
    let rows = total.parent().and_downcast::<gtk::ListBox>().unwrap().observe_children();
    let n = rows.n_items();
    if utils::check_date(hasta) && hasta != "" {
        let mut tot = Decimal::new(000, 2);
        let f_db = utils::convert_fech(hasta);
        for i in 0..n-1 {
            let _ = conn.query_row(&format!("SELECT SUM(importe) FROM '{}' WHERE fecha <= ?", sl.string(i).unwrap()), [&f_db], |row| {
                let saldo: f64 = row.get(0).unwrap_or_else(|_| { 0.0 });
                let saldo = Decimal::from_f64(saldo).unwrap();
                tot += saldo;
                let s = utils::format_euros(saldo);
                rows.item(i).and_downcast::<adw::ActionRow>().unwrap().set_subtitle(&s);
                Ok(())
            });
        }
        let t = utils::format_euros(tot);
        total.set_subtitle(&t);
    } else {
        for i in 0..n-1 {
            rows.item(i).and_downcast::<adw::ActionRow>().unwrap().set_subtitle("-");
        }
        total.set_subtitle("-");
    }
}

fn cajas_saldos(conn: &Rc<rusqlite::Connection>, desde: &str, hasta: &str, caja: &adw::ComboRow, entradas: &adw::ActionRow, salidas: &adw::ActionRow, total: &adw::ActionRow) {
    if utils::check_date(desde) && desde != "" && utils::check_date(hasta) && hasta != "" {
        let caja_str = caja.selected_item().unwrap().downcast::<gtk::StringObject>().unwrap().string();
        let desde_db = utils::convert_fech(desde);
        let hasta_db = utils::convert_fech(hasta);
        let mut tot = Decimal::new(000, 2);
        let _ = conn.query_row(&format!("SELECT SUM(importe) FROM '{}' WHERE importe > 0 AND fecha BETWEEN ?1 AND ?2", caja_str), [&desde_db, &hasta_db], |row| {
            let ent: f64 = row.get(0).unwrap_or_else(|_| { 0.0 });
            let ent = Decimal::from_f64(ent).unwrap();
            tot += ent;
            let s = utils::format_euros(ent);
            entradas.set_subtitle(&s);
            Ok(())
        });
        let _ = conn.query_row(&format!("SELECT SUM(importe) FROM '{}' WHERE importe < 0 AND fecha BETWEEN ?1 AND ?2", caja_str), [&desde_db, &hasta_db], |row| {
            let sal: f64 = row.get(0).unwrap_or_else(|_| { 0.0 });
            let sal = Decimal::from_f64(sal).unwrap();
            tot += sal;
            let s = utils::format_euros(sal);
            salidas.set_subtitle(&s);
            Ok(())
        });
        let s = utils::format_euros(tot);
        total.set_subtitle(&s);
    } else {
        entradas.set_subtitle("-");
        salidas.set_subtitle("-");
        total.set_subtitle("-");
    }
}

fn etiquetas_saldos(conn: &Rc<rusqlite::Connection>, desde: &str, hasta: &str, et: &adw::ComboRow, entradas: &adw::ActionRow, salidas: &adw::ActionRow, total: &adw::ActionRow, sl: &gtk::StringList) {
    if utils::check_date(desde) && desde != "" && utils::check_date(hasta) && hasta != "" {
        let desde_db = utils::convert_fech(desde);
        let hasta_db = utils::convert_fech(hasta);
        let mut tot = Decimal::new(000, 2);
        let et_str = et.selected_item().unwrap().downcast::<gtk::StringObject>().unwrap().string();
        let etq = format!("(etiquetas = '{0}' OR etiquetas LIKE '{0} |%' OR etiquetas LIKE '%| {0}' OR etiquetas LIKE '%| {0} |%')", et_str);
        let mut query = String::from("SELECT ");
        for i in 0..sl.n_items() {
            query.push_str(&format!("SUM(importe) FROM '{}' WHERE importe > 0 AND {} AND fecha BETWEEN '{}' AND '{}'", sl.string(i).unwrap(), etq, desde_db, hasta_db));
            if i != sl.n_items()-1 {
                query.push_str(" UNION ALL SELECT ");
            }
        }
        let _ = conn.query_row(&query, (), |row| {
            let ent: f64 = row.get(0).unwrap_or_else(|_| { 0.0 });
            let ent = Decimal::from_f64(ent).unwrap();
            tot += ent;
            let s = utils::format_euros(ent);
            entradas.set_subtitle(&s);
            Ok(())
        });
        
        let mut query = String::from("SELECT ");
        for i in 0..sl.n_items() {
            query.push_str(&format!("SUM(importe) FROM '{}' WHERE importe < 0 AND {} AND fecha BETWEEN '{}' AND '{}'", sl.string(i).unwrap(), etq, desde_db, hasta_db));
            if i != sl.n_items()-1 {
                query.push_str(" UNION ALL SELECT ");
            }
        }
        let _ = conn.query_row(&query, (), |row| {
            let sal: f64 = row.get(0).unwrap_or_else(|_| { 0.0 });
            let sal = Decimal::from_f64(sal).unwrap();
            tot += sal;
            let s = utils::format_euros(sal);
            salidas.set_subtitle(&s);
            Ok(())
        });
        let s = utils::format_euros(tot);
        total.set_subtitle(&s);
        
    } else {
        entradas.set_subtitle("-");
        salidas.set_subtitle("-");
        total.set_subtitle("-");
    }
}
