use gtk::glib;
use std::rc::Rc;

pub const DB_REL_DIR: &str = "/.local/share/tesoreria/";
pub const DB_FILE: &str = "tesoreria.db";

pub fn exist_db() -> bool {
    let path_dir = glib::home_dir().into_os_string().into_string().unwrap() + DB_REL_DIR;
    let _ = std::fs::create_dir(&path_dir);
    let path = path_dir + DB_FILE;
    std::path::Path::new(&path).exists()
}

pub fn state_cif_db() -> (rusqlite::Connection, bool) {
    let path = glib::home_dir().into_os_string().into_string().unwrap() + DB_REL_DIR + DB_FILE;
    let conn = rusqlite::Connection::open(&path).unwrap();
    let r = test_conn(&conn);
    (conn, !r)
}

pub fn test_conn(conn: &rusqlite::Connection) -> bool {
    if conn.query_row("SELECT name FROM sqlite_schema", [], |_| Ok(())).is_err() {
        false
    } else {
        true
    }
}

pub fn conn_db(pass: &str) -> rusqlite::Connection {
    let path = glib::home_dir().into_os_string().into_string().unwrap() + DB_REL_DIR + DB_FILE;
    let conn = rusqlite::Connection::open(&path).unwrap();
    if pass != "" {
        let _ = conn.pragma_update(Some(rusqlite::DatabaseName::Main),"key", pass);
    }
    conn
}

pub fn export_db(conn: &Rc<rusqlite::Connection>, pass: &str) -> rusqlite::Connection {
    let path_aux = glib::home_dir().into_os_string().into_string().unwrap() + DB_REL_DIR + "aux.db";
    let _ = conn.execute("ATTACH DATABASE ?1 AS aux KEY ?2", [&path_aux, pass]);
    let _ = conn.execute("SELECT sqlcipher_export('aux')", []);
    let _ = conn.execute("DETACH DATABASE aux", []);
    let path = glib::home_dir().into_os_string().into_string().unwrap() + DB_REL_DIR + DB_FILE;
    let _ = std::fs::remove_file(&path);
    let _ = std::fs::rename(&path_aux, &path);
    let new_conn = conn_db(&pass).into();
    new_conn
}

pub fn crear_tablas(conn: &rusqlite::Connection) {
    conn.execute(
        "CREATE TABLE afiliacion (
            cc          TEXT PRIMARY KEY,
            apellidos   TEXT,
            nombre      TEXT,
            mcotiza     TEXT,
            cuota       REAL,
            fnacimiento TEXT,
            telefono    TEXT,
            email       TEXT,
            dir         TEXT,
            loc         TEXT,
            barrio      TEXT,
            obs         TEXT,
            ocu         TEXT,
            sec         TEXT,
            centro      TEXT,
            dirc        TEXT,
            obsc        TEXT,
            afiliado    TEXT,
            fechabaja   TEXT
        )",
        (),
    ).unwrap();
    conn.execute(
        "CREATE TABLE cotizaciones (
            cc          TEXT,
            num         INTEGER,
            caja        TEXT,
            mes         TEXT
        )",
        (),
    ).unwrap();
    conn.execute(
        "CREATE TABLE etiquetas (
            nombre      TEXT PRIMARY KEY
        )",
        (),
    ).unwrap();
    conn.execute("INSERT INTO etiquetas (nombre) VALUES ('Cotización')", ()).unwrap();
}

pub fn lista_cajas(conn: &Rc<rusqlite::Connection>) -> gtk::StringList {
    let sl = gtk::StringList::new(&[]);
    let mut stmt = conn.prepare("SELECT name FROM sqlite_schema WHERE type ='table' AND name!='cotizaciones' AND name!='afiliacion' AND name!='etiquetas'").unwrap();
    let mut rows = stmt.query([]).unwrap();
    while let Some(row) = rows.next().unwrap() {
        let m: String = row.get(0).unwrap();
        sl.append(&m);
    }
    sl
}

pub fn futuro(conn: &Rc<rusqlite::Connection>, caja: &str) -> bool {
    if conn.query_row(&format!("SELECT fecha FROM '{}' WHERE fecha > date('now', 'localtime')", caja), [], |_| Ok(())).is_err() {
        false
    } else {
        true
    }
}

pub fn es_v1(conn: &rusqlite::Connection) -> bool {
    let mut v1 = 0;
    let mut stmt = conn.prepare("SELECT name FROM sqlite_schema WHERE type ='table'").unwrap();
    let mut rows = stmt.query([]).unwrap();
    while let Some(row) = rows.next().unwrap() {
        let m: String = row.get(0).unwrap();
        if m == "altas" || m == "bajas" || m == "cotizaciones" {
            v1 += 1;
        }
    }
    if v1 == 3 {
        true
    } else {
        false
    }
}

struct Cot {
    cc: String,
    cc_obs: String,
    cotizado: String,
    meses: u32,
    proxcot: String,
}

pub fn v1_v2(conn: &rusqlite::Connection) {
    let tx = conn.unchecked_transaction().unwrap();
    {
        let _ = tx.execute("DROP TABLE cotizaciones", ());
        let mut vcajas: Vec<String> = Vec::new();
        let mut stmt = tx.prepare("SELECT name FROM sqlite_schema WHERE type ='table' AND name!='altas' AND name!='bajas'").unwrap();
        let mut rows = stmt.query([]).unwrap();
        while let Some(row) = rows.next().unwrap() {
            let m: String = row.get(0).unwrap();
            vcajas.push(m);
        } 
        
        crear_tablas(&tx);
        let mut vcots: Vec<Cot> = Vec::new();
        let mut stmt = tx.prepare("SELECT cc, obs, cotizado, nombre, apellido1, apellido2, fnacimiento, telefono, email, dir, zona, ocu, sec, centro, dirc, obsc, afiliado, '' FROM altas UNION ALL SELECT cc, obs, cotizado, nombre, apellido1, apellido2, fnacimiento, telefono, email, dir, zona, ocu, sec, centro, dirc, obsc, afiliado, fechabaja FROM bajas").unwrap();
        let mut rows = stmt.query([]).unwrap();
        while let Some(row) = rows.next().unwrap() {
            let cc: String = row.get(0).unwrap();
            let obs: String = row.get(1).unwrap_or_else(|_| "".to_string());
            let obsv: Vec<_> = obs.split_whitespace().collect();
            let cc_obs = if obsv.len() == 3 && obsv[0] == "C.C." && obsv[1] == "antiguo:" {
                let mut chs =  obsv[2].chars();
                chs.next_back();
                chs.as_str()
            } else {
                ""
            };
            let cotizado: String = row.get(2).unwrap_or_else(|_| "".to_string());
            
            let nombre: String = row.get(3).unwrap_or_else(|_| "".to_string());
            let apellido1: String = row.get(4).unwrap_or_else(|_| "".to_string());
            let apellido2: String = row.get(5).unwrap_or_else(|_| "".to_string());
            let fnacimiento: String = row.get(6).unwrap_or_else(|_| "".to_string());
            let telefono: String = row.get(7).unwrap_or_else(|_| "".to_string());
            let email: String = row.get(8).unwrap_or_else(|_| "".to_string());
            let dir: String = row.get(9).unwrap_or_else(|_| "".to_string());
            let zona: String = row.get(10).unwrap_or_else(|_| "".to_string());
            let ocu: String = row.get(11).unwrap_or_else(|_| "".to_string());
            let sec: String = row.get(12).unwrap_or_else(|_| "".to_string());
            let centro: String = row.get(13).unwrap_or_else(|_| "".to_string());
            let dirc: String = row.get(14).unwrap_or_else(|_| "".to_string());
            let obsc: String = row.get(15).unwrap_or_else(|_| "".to_string());
            let afiliado: String = row.get(16).unwrap_or_else(|_| "".to_string());
            let fechabaja: String = row.get(17).unwrap_or_else(|_| "".to_string());
            
            let apellidos = if apellido2 == "" { apellido1 } else { format!("{} {}", apellido1, apellido2) };
            
            tx.execute(
                "INSERT INTO afiliacion (cc, apellidos, nombre, mcotiza, cuota, fnacimiento, telefono, email, dir, loc, barrio, obs, ocu, sec, centro, dirc, obsc, afiliado, fechabaja) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)",
                [&cc, &apellidos, &nombre, "", "10.0", &fnacimiento, &telefono, &email, &dir, &zona, "", &obs, &ocu, &sec, &centro, &dirc, &obsc, &afiliado, &fechabaja],
            ).unwrap();
            
            vcots.push(Cot{cc, cc_obs: cc_obs.to_string(), meses: 0, cotizado, proxcot: "".to_string()});
        }
        let _ = tx.execute("DROP TABLE altas", ());    
        let _ = tx.execute("DROP TABLE bajas", ());
        
        if vcajas.len() > 0 {
            let mut query = String::from("SELECT ");
            for i in 0..vcajas.len() {
                query.push_str(&format!("concepto, fecha FROM '{}' WHERE concepto LIKE '% cotiza % mes%'", vcajas[i]));
                if i != vcajas.len()-1 {
                    query.push_str(" UNION ALL SELECT ");
                }
            }
            query.push_str(" ORDER BY fecha");
            let mut stmt = tx.prepare(&query).unwrap();
            let mut rows = stmt.query([]).unwrap();
            while let Some(row) = rows.next().unwrap() {
                let conc: String = row.get(0).unwrap();
                let p: Vec<_> = conc.split_whitespace().collect();
                for i in 0..vcots.len() {
                    if vcots[i].cc == p[0] || vcots[i].cc_obs == p[0] {
                        vcots[i].meses += p[2].parse::<u32>().unwrap();
                        if vcots[i].proxcot == "" {
                            let fecha: String = row.get(1).unwrap();
                            vcots[i].proxcot = format!("{}{}", &fecha[..4], &fecha[5..7]);
                        }
                        break;
                    }
                }
            }
        
            for cot in &mut vcots {
                if cot.proxcot != "" && cot.meses != 0 {
                    cot.proxcot = resta_meses_v1_v2(&cot.cotizado, cot.meses);
                }
            }
            
            for caja in &vcajas {
                let _ = tx.execute(&format!(
                    "CREATE TABLE '{}_v2' (
                        num         INTEGER PRIMARY KEY,
                        fecha       DATE,
                        factura     TEXT,
                        concepto    TEXT,
                        etiquetas   TEXT,
                        importe     REAL
                    )", caja),
                    ());
            }
            
            let mut query = String::from("SELECT ");
            for i in 0..vcajas.len() {
                query.push_str(&format!("'{0}' AS caja, fecha, factura, concepto, euros FROM '{}'", vcajas[i]));
                if i != vcajas.len()-1 {
                    query.push_str(" UNION ALL SELECT ");
                }
            }
            query.push_str("ORDER BY fecha");
            let mut stmt = tx.prepare(&query).unwrap();
            let mut rows = stmt.query([]).unwrap();
            while let Some(row) = rows.next().unwrap() {
                let caja: String = row.get(0).unwrap();
                let caja_v2 = format!("{}_v2", caja);
                let mut fecha: String = row.get(1).unwrap();
                fecha = fecha.split_whitespace().next().unwrap().to_string();
                let factura: String = row.get(2).unwrap_or_else(|_| "".to_string());
                let concepto: String = row.get(3).unwrap_or_else(|_| "".to_string());
                let p: Vec<_> = concepto.split_whitespace().collect();
                let et = if p.len() == 4 && p[1] == "cotiza" {
                    "Cotización"
                } else {
                    ""
                };
                let euros: f32 = row.get(4).unwrap();
                let _ = tx.execute(&format!("INSERT INTO '{}' (fecha, factura, concepto, etiquetas, importe) VALUES (?, ?, ?, ?, ?)", caja_v2), [&fecha, &factura, &concepto, et,  &euros.to_string()]);
                
                if et == "Cotización" {
                    let num_insert = tx.last_insert_rowid();
                    for cot in &mut vcots {
                        if cot.cc == p[0] || cot.cc_obs == p[0] {
                            let meses = p[2].parse::<u32>().unwrap();
                            for _ in 0..meses {
                                cot.proxcot = suma_mes_v1_v2(&cot.proxcot);
                                let _ = tx.execute("INSERT INTO cotizaciones (cc, num, caja, mes) VALUES (?, ?, ?, ?)", [&cot.cc, &num_insert.to_string(), &caja, &cot.proxcot]);
                            }
                            break;
                        }
                    }
                }
            }
            
            for caja in vcajas {
                let _ = tx.execute(&format!("DROP TABLE '{}'", caja), ());
                let _ = tx.execute(&format!("ALTER TABLE '{0}_v2' RENAME TO '{}'", caja), ());
            }
        }
    }
    let _ = tx.commit();
}

fn resta_meses_v1_v2(cotizado: &str, meses: u32) -> String {
    let mut anyo = cotizado[..4].parse::<i32>().unwrap();
    let mut mes = cotizado[5..7].parse::<i32>().unwrap();
    let resto = meses as i32 % 12;
    let anyos = meses as i32 / 12;
    anyo = anyo - anyos;
    mes = mes - resto;
    if mes <= 0 {
        mes = mes + 12;
        anyo = anyo - 1;
    }
    format!("{}{:02}", anyo, mes)
}

fn suma_mes_v1_v2(val: &str) -> String {
    let mut anyo = val[..4].parse::<i32>().unwrap();
    let mut mes = val[4..6].parse::<i32>().unwrap();
    if mes != 12 {
        mes += 1;
    } else {
        mes = 1;
        anyo += 1;
    }
    format!("{}{:02}", anyo, mes)
}
