use adw::prelude::*;
use gtk::glib;
use std::rc::Rc;
use crate::db_utils;
use crate::utils;
use crate::pages_ca;
use crate::pages_af;
use crate::ie_csvs;

pub const COLUMNAS: &[&str] = &["Número", "Fecha", "Factura", "Concepto", "Etiquetas", "Importe", "Saldo"];

pub fn ca_prev(conn: &Rc<rusqlite::Connection>, stack: &adw::ViewStack, nav: &adw::NavigationView, window: &adw::ApplicationWindow, searchentry: &gtk::SearchEntry, store: &gtk::gio::ListStore, store_af: &gtk::gio::ListStore) {
    let crearb= gtk::Button::builder()
        .halign(gtk::Align::Center)
        .css_classes(["pill", "suggested-action"])
        .label("Crear caja")
        .build();
    let sp = adw::StatusPage::builder()
        .title("No existe ninguna caja")
        .icon_name("folder-symbolic")
        .child(&crearb)
        .vexpand(true)
        .build();
    stack.add_titled_with_icon(&sp, Some("ca"), "Cajas", "view-list-symbolic");
    let conn_clone = conn.clone();
    let nav_clone = nav.clone();
    let stack_clone = stack.clone();
    let window_clone = window.clone();
    let searchentry_clone = searchentry.clone();
    let store_clone = store.clone();
    let store_af_clone = store_af.clone();
    crearb.connect_clicked(move |_| {
        pages_ca::crear_caja(&conn_clone, &nav_clone, Some(&stack_clone), &window_clone, Some(&searchentry_clone), None, Some(&store_clone), Some(&store_af_clone));
    });
}

pub fn ca(conn: &Rc<rusqlite::Connection>, stack: &adw::ViewStack, nav: &adw::NavigationView, window: &adw::ApplicationWindow, searchentry: &gtk::SearchEntry, store: &gtk::gio::ListStore, store_af: &gtk::gio::ListStore) {
    let settings = gtk::gio::Settings::new("org.cntait.tesoreria.cajas");
    let vbox = gtk::Box::new(gtk::Orientation::Vertical, 0);      
    let cbox = gtk::CenterBox::builder()
        .margin_top(10)
        .margin_bottom(10)
        .build();
    let sel = settings.int("caja");
    let sl_cajas = db_utils::lista_cajas(&conn);
    let dd = gtk::DropDown::builder()
        .model(&sl_cajas)
        .selected(sel as u32)
        .tooltip_text("Seleccionar caja")
        .halign(gtk::Align::Center)
        .hexpand(true)
        .build();
    cbox.set_start_widget(Some(&dd));
    let now = glib::DateTime::now_local().unwrap();
    let today = format!("{:02}/{:02}/{}" ,now.day_of_month(), now.month(), now.year());
    let mut meses_ant = settings.int("meses");
    if meses_ant > 120 {
        meses_ant = 120;
    }
    let desde_init = now.add_months(-meses_ant).unwrap();
    let desde_init_str = format!("{:02}/{:02}/{}" ,desde_init.day_of_month(), desde_init.month(), desde_init.year());
    let desde = utils::entry_calendar(&desde_init_str);
    let hasta = utils::entry_calendar(&today);
    let hboxc = utils::crea_hb("Desde:", &desde, "Hasta:", &hasta);
    cbox.set_center_widget(Some(&hboxc));
    let filtrosb = gtk::MenuButton::builder()
        .label("Filtros")
        .tooltip_text("Seleccionar filtros")
        .halign(gtk::Align::Center)
        .build();
    filtrosb.set_hexpand(true);
    cbox.set_end_widget(Some(&filtrosb));
    vbox.append(&cbox);
    let columnview = gtk::ColumnView::builder()
        .reorderable(false)
        .build();
    let scrolled_window = gtk::ScrolledWindow::builder()
        .child(&columnview)
        .vexpand(true)
        .build();
    vbox.append(&scrolled_window);
    let actbar = gtk::ActionBar::new();
    let mca = gtk::MenuButton::new();
    actbar.pack_start(&mca);
    let nf = gtk::Label::new(None);
    actbar.set_center_widget(Some(&nf));
    let cont_mov = adw::ButtonContent::builder()
        .icon_name("tab-new-symbolic")
        .label("Movimiento")
        .build();
    let mov = gtk::Button::builder()
        .child(&cont_mov)
        .css_classes(["suggested-action"])
        .build();
    actbar.pack_end(&mov);
    let cont_sald = adw::ButtonContent::builder()
        .icon_name("view-grid-symbolic")
        .label("Saldos")
        .build();
    let sald = gtk::Button::builder()
        .child(&cont_sald)
        .css_classes(["raised"])
        .build();
    actbar.pack_end(&sald);
    vbox.append(&actbar);
    stack.add_titled_with_icon(&vbox, Some("ca"), "Cajas", "view-list-symbolic");
    
    let conn_clone = conn.clone();
    let desde_clone = desde.clone();
    let hasta_clone = hasta.clone();
    let filtrosb_clone = filtrosb.clone();
    let searchentry_clone = searchentry.clone();
    let custfilter = gtk::CustomFilter::new(move |item| {
        filtros(&item, &desde_clone, &hasta_clone, &filtrosb_clone, &searchentry_clone)
    });
    let fstore = gtk::FilterListModel::new(Some(store.clone()), Some(custfilter.clone()));
    let sel = gtk::SingleSelection::new(Some(fstore.clone()));
    sel.set_autoselect(false);
    sel.set_can_unselect(true);
    columnview.set_model(Some(&sel));
    let wb = adw::WrapBox::new();
    wb.set_natural_line_length(200); //Valor por defecto (si hay una etiqueta más larga se modifica)
    wb.set_child_spacing(5);
    wb.set_line_spacing(5);
    
    let pop_ctx = crea_menu_contextual(&conn_clone, &nav, &window, &store, &sel, &wb, &custfilter, &store_af);
    pop_ctx.set_parent(&columnview);
    pop_ctx.connect_closed(|pop_ctx| {
        let menu_ctx = pop_ctx.menu_model().and_downcast::<gtk::gio::Menu>().unwrap();
        if menu_ctx.n_items() > 2 {
            menu_ctx.remove(2);
        }
    });
    crea_columnas(&columnview, &sel, &pop_ctx);
    let nf_clone = nf.clone();
    fstore.connect_items_changed(move |fstore,_,_,_| {
        nfilas(&fstore, &nf_clone);
    });
    crea_menu_filtros(&filtrosb, &custfilter, &columnview, &conn, &store, &wb);
    crea_menu(&mca, &conn_clone, &nav, &window, &store, &store_af, &dd, &wb, &custfilter, &stack, &searchentry, &fstore, &columnview);
    crea_lista(&dd, &store, &conn);
    if fstore.n_items() > 0 {
        columnview.scroll_to(0, None, gtk::ListScrollFlags::NONE, None);
    }
    let gc = gtk::GestureClick::new();
    gc.set_button(gtk::gdk::BUTTON_PRIMARY);
    columnview.add_controller(gc.clone());
    let columnview_clone = columnview.clone();
    gc.connect_pressed(move |_, _, _, _| {
        sel.unselect_all();
        columnview_clone.grab_focus();
    });
    let conn_clone = conn.clone();
    let settings_clone = settings.clone();
    let store_clone = store.clone();
    let columnview_clone = columnview.clone();
    dd.connect_selected_notify(move |dd| {
        crea_lista(&dd, &store_clone, &conn_clone);
        if fstore.n_items() > 0 {
            columnview_clone.scroll_to(0, None, gtk::ListScrollFlags::NONE, None);
        }
        let _ = settings_clone.set_int("caja", dd.selected() as i32);
    });
    let custfilter_clone = custfilter.clone();
    let hasta_clone = hasta.clone();
    let settings_clone = settings.clone();
    desde.connect_changed(move |desde| {
        custfilter_clone.changed(gtk::FilterChange::Different);
        let dif = dif_meses(&desde.text(), &hasta_clone.text());
        if dif > 1 {
            let _ = settings_clone.set_int("meses", dif as i32);
        } else {
            let _ = settings_clone.set_int("meses", 1);
        }
    });
    let custfilter_clone = custfilter.clone();
    let desde_clone = desde.clone();
    hasta.connect_changed(move |hasta| {
        custfilter_clone.changed(gtk::FilterChange::Different);
        let dif = dif_meses(&desde_clone.text(), &hasta.text());
        if dif > 1 {
            let _ = settings.set_int("meses", dif as i32);
        } else {
            let _ = settings.set_int("meses", 1);
        }
    });
    let custfilter_clone = custfilter.clone();
    let columnview_clone = columnview.clone();
    searchentry.connect_search_changed(move |searchentry| {
        let col_saldo = columnview_clone.columns().item(6).and_downcast::<gtk::ColumnViewColumn>().unwrap();
        if searchentry.text() == "" {
            col_saldo.set_visible(true);
        } else {
            col_saldo.set_visible(false);
        }
        custfilter_clone.changed(gtk::FilterChange::Different);
    });
    let nav_clone = nav.clone();
    let conn_clone = conn.clone();
    sald.connect_clicked(move |_| {
        pages_ca::saldos(&conn_clone, &nav_clone);
    });
    let window_clone = window.clone();
    let nav_clone = nav.clone();
    let store_clone = store.clone();
    let conn_clone = conn.clone();
    mov.connect_clicked(move |_| {
        pages_ca::mov(&conn_clone, &nav_clone, &window_clone, &store_clone, &wb, &custfilter, None, Some(&columnview));
    });
}

fn crea_columnas(columnview: &gtk::ColumnView, sel: &gtk::SingleSelection, pop: &gtk::PopoverMenu) {
    let cn = COLUMNAS;
    for i in 0..cn.len() {
        let colfactory = gtk::SignalListItemFactory::new();
        let sel_clone = sel.clone();
        let pop_clone = pop.clone();
        let columnview_clone = columnview.clone();
        colfactory.connect_setup(move |_factory, item| {
            let item = item.downcast_ref::<gtk::ListItem>().unwrap();
            let label = gtk::Label::new(None);
            label.set_halign(gtk::Align::Start);
            let cbox = gtk::Box::new(gtk::Orientation::Horizontal, 0);
            cbox.append(&label);
            item.set_child(Some(&cbox));
            
            let gc = gtk::GestureClick::new();
            gc.set_button(gtk::gdk::BUTTON_SECONDARY);
            cbox.add_controller(gc.clone());
            let item_clone = item.clone();
            let sel_clone = sel_clone.clone();
            let pop_clone = pop_clone.clone();
            let columnview_clone = columnview_clone.clone();
            gc.connect_pressed(move |_, _, x, y| {
                let pres = item_clone.item().unwrap();
                let model = sel_clone.model().unwrap();
                for i in 0..model.n_items() {
                    if pres == model.item(i).unwrap() {
                        sel_clone.select_item(i, true);
                        break;
                    }
                }
                let npo = cbox.compute_point(&columnview_clone, &gtk::graphene::Point::new(x as f32, y as f32));
                pop_clone.set_pointing_to(Some(&gtk::gdk::Rectangle::new(npo.unwrap().x() as i32, npo.unwrap().y() as i32, 0, 0)));
                if sel_clone.selected_item().and_downcast::<gtk::StringList>().unwrap().string(4).unwrap() == "Cotización" {
                    let menu_ctx = pop_clone.menu_model().and_downcast::<gtk::gio::Menu>().unwrap();
                    menu_ctx.insert(2, Some("Devolver cuota"), Some("win.Devolvercuota_ca_ctx"));
                }
                pop_clone.popup();
            });
        });
        colfactory.connect_bind(move |_, item| {
            let item = item.downcast_ref::<gtk::ListItem>().unwrap();
            let cbox = item.child().and_downcast::<gtk::Box>().unwrap();
            let label = cbox.first_child().and_downcast::<gtk::Label>().unwrap();
            let strl = item.item().and_downcast::<gtk::StringList>().unwrap();
            let strl_n = strl.string(i.try_into().unwrap()).unwrap();
            label.set_label(&strl_n);
        });
        let col = gtk::ColumnViewColumn::new(Some(&cn[i]), Some(colfactory));
        columnview.append_column(&col);
        if i == 3 { col.set_expand(true) } // Concepto
    }
}

fn filtros(item: &glib::Object, desde: &gtk::Entry, hasta: &gtk::Entry, filtros: &gtk::MenuButton, searchentry: &gtk::SearchEntry) -> bool {
    let stl = item.downcast_ref::<gtk::StringList>().unwrap();
    
    // Corresponde con la primera fila en la que solo se guarda en string(0) el nombre de la caja
    if stl.string(1) == None { return false; }
    // Filtro fechas
    if !utils::check_date(&desde.text()) || desde.text() == "" || !utils::check_date(&hasta.text()) || hasta.text() == "" { return false; }
    let desde = utils::convert_fech(&desde.text());
    let hasta = utils::convert_fech(&hasta.text());
    let fecha = utils::convert_fech(&stl.string(1).unwrap());
    if fecha < desde || fecha > hasta { return false; }
    // Filtro entradas o salidas
    let mut ch = filtros.popover().unwrap().child().unwrap().first_child().unwrap();
    if !ch.downcast_ref::<gtk::CheckButton>().unwrap().is_active() {
        ch = ch.next_sibling().unwrap();
        if ch.downcast_ref::<gtk::CheckButton>().unwrap().is_active() { 
            if stl.string(5).unwrap().chars().nth(0).unwrap() == '-' || stl.string(5).unwrap() == "0,00 €" { return false; }
        }
        ch = ch.next_sibling().unwrap();
        if ch.downcast_ref::<gtk::CheckButton>().unwrap().is_active() { 
            if stl.string(5).unwrap().chars().nth(0).unwrap() != '-' { return false; }
        }
    }
    // Filtro búsqueda
    let b = searchentry.text();
    if b != "" {
        let mut aux = false;
        for i in 0..stl.n_items() {
            if stl.string(i).unwrap().to_lowercase().contains(&b.to_lowercase()) {
                aux = true;
                break;
            }
        }
        if !aux { return false; }
    }
    // Filtros etiquetas
    let wb = filtros.popover().unwrap().child().unwrap().last_child().and_downcast::<adw::WrapBox>().unwrap();
    if wb.is_sensitive() {
        let mut algo = false;
        let mut aux = false;
        let chs = wb.observe_children();
'fuera: for i in 0..chs.n_items() {
            let tg = chs.item(i).and_downcast::<gtk::ToggleButton>().unwrap();
            if tg.is_active() {
                algo = true;
                let ets = stl.string(4).unwrap();
                let mut iter = ets.split(" | ");
                while let Some(et) = iter.next() {
                    if et == tg.label().unwrap() {
                        aux = true;
                        break 'fuera;
                    }
                }
            }
        }
        if algo && !aux { return false; }
    }
    true
}

fn crea_menu_contextual(conn: &Rc<rusqlite::Connection>, nav: &adw::NavigationView, window: &adw::ApplicationWindow, store: &gtk::gio::ListStore, sel: &gtk::SingleSelection, wb: &adw::WrapBox, custfilter: &gtk::CustomFilter, store_af: &gtk::gio::ListStore) -> gtk::PopoverMenu {
    let ops = ["Editar", "Eliminar", "Devolver cuota"];
    let menu = gtk::gio::Menu::new();
    for op in ops {
        let action_name: String = op.chars().filter(|&c| !c.is_whitespace()).collect();
        let action = gtk::gio::SimpleAction::new(&format!("{}_ca_ctx", action_name), None);
        window.add_action(&action);
        if op == "Editar" {
            menu.append(Some(&op), Some(&format!("win.{}_ca_ctx", action_name)));
            let conn_clone = conn.clone();
            action.connect_activate(glib::clone!(
                #[weak]
                nav,
                #[weak]
                window,
                #[weak]
                store,
                #[weak]
                store_af,
                #[weak]
                sel,
                #[weak]
                wb,
                #[weak]
                custfilter,
                move |_, _| {
                    let pos = sel.selection().minimum();
                    let binding = sel.item(pos).unwrap();
                    let osl = binding.downcast_ref::<gtk::StringList>();
                    if osl.unwrap().string(4).unwrap() == "Cotización" {
                       pages_af::cotizar(&conn_clone, &nav, &window, &store_af, None, &store, osl);
                    } else {
                        pages_ca::mov(&conn_clone, &nav, &window, &store, &wb, &custfilter, osl, None);
                    }
                }
            ));
        } else if op == "Eliminar" {
            menu.append(Some(&op), Some(&format!("win.{}_ca_ctx", action_name)));
            let conn_clone = conn.clone();
            action.connect_activate(glib::clone!(
                #[weak]
                window,
                #[weak]
                store,
                #[weak]
                sel,
                #[weak]
                store_af,
                move |_, _| {
                    let sl = sel.selected_item().and_downcast::<gtk::StringList>().unwrap();
                    if sl.string(4).unwrap() == "Cotización" {
                        pages_ca::eliminar_cot(&conn_clone, &window, &store, &sl, &store_af);
                    } else {
                        pages_ca::eliminar_mov(&conn_clone, &window, &store, &sl);
                    }
                }
            ));
        } else if op == "Devolver cuota" {
            let conn_clone = conn.clone();
            action.connect_activate(glib::clone!(
                #[weak]
                nav,
                #[weak]
                window,
                #[weak]
                store,
                #[weak]
                store_af,
                #[weak]
                sel,
                move |_, _| {
                    pages_ca::devolver(&conn_clone, &nav, &window, &store, &store_af, &sel); 
                }
            ));
        }
    }
    gtk::PopoverMenu::builder()
        .menu_model(&menu)
        .has_arrow(false)
        .halign(gtk::Align::Start)
        .build()
}

fn nfilas(fstore: &gtk::FilterListModel, nf: &gtk::Label) {
    let nitems = fstore.n_items();
    let plural = if nitems == 1 { "" } else { "s" };
    nf.set_label(&format!("{} fila{}", nitems, plural));
}

fn crea_menu_filtros(filtros: &gtk::MenuButton, custfilter: &gtk::CustomFilter, cv: &gtk::ColumnView, conn: &Rc<rusqlite::Connection>, store: &gtk::gio::ListStore, wb: &adw::WrapBox){
    let col_saldo = cv.columns().item(6).and_downcast::<gtk::ColumnViewColumn>().unwrap();
    let vbox = gtk::Box::new(gtk::Orientation::Vertical, 0);
    let todos = gtk::CheckButton::with_label("Todos los movimientos");
    todos.set_active(true);
    vbox.append(&todos);
    let entradas = gtk::CheckButton::with_label("Entradas");
    entradas.set_group(Some(&todos));
    let custfilter_clone = custfilter.clone();
    let col_saldo_clone = col_saldo.clone();
    entradas.connect_toggled(move |_| {
        col_saldo_clone.set_visible(false);
        custfilter_clone.changed(gtk::FilterChange::Different);
    });
    vbox.append(&entradas);
    let salidas = gtk::CheckButton::with_label("Salidas");
    salidas.set_group(Some(&todos));
    let custfilter_clone = custfilter.clone();
    let col_saldo_clone = col_saldo.clone();
    salidas.connect_toggled(move |_| {
        col_saldo_clone.set_visible(false);
        custfilter_clone.changed(gtk::FilterChange::Different);
    });
    vbox.append(&salidas);
    let sep = gtk::Separator::new(gtk::Orientation::Horizontal);
    vbox.append(&sep);
    let hbox = gtk::Box::new(gtk::Orientation::Horizontal, 2); //builder
    hbox.set_margin_top(5);
    hbox.set_margin_bottom(5);
    hbox.set_margin_start(5);
    hbox.set_margin_end(5);
    let lab = gtk::Label::new(Some("Etiquetas")); //builder
    lab.set_halign(gtk::Align::Start);
    lab.set_hexpand(true);
    hbox.append(&lab);
    let gest = gtk::Button::from_icon_name("document-edit-symbolic");
    gest.add_css_class("flat");
    hbox.append(&gest);
    let sw = gtk::Switch::new();
    sw.set_valign(gtk::Align::Center);
    hbox.append(&sw);
    vbox.append(&hbox);
    wb.set_sensitive(false);
    let mut stmt = conn.prepare("SELECT nombre FROM etiquetas").unwrap();
    let mut rows = stmt.query([]).unwrap();
    while let Some(row) = rows.next().unwrap() {
        let et: String = row.get(0).unwrap();
        let tg = gtk::ToggleButton::with_label(&et);
        wb.append(&tg);
        let tg_width = tg.measure(gtk::Orientation::Horizontal, -1).0;
        if wb.natural_line_length() < tg_width {
            wb.set_natural_line_length(tg_width);
        }
        let custfilter_clone = custfilter.clone();
        tg.connect_toggled(move |_| {
            custfilter_clone.changed(gtk::FilterChange::Different);
        });
    }
    vbox.append(wb);
    let pop = gtk::Popover::builder()
        .child(&vbox)
        .has_arrow(false)
        .autohide(true)
        .build();
    pop.set_offset(0, 5);
    filtros.set_popover(Some(&pop));

    let custfilter_clone = custfilter.clone();
    let col_saldo_clone = col_saldo.clone();
    let sw_clone = sw.clone();
    todos.connect_toggled(move |_| {
        custfilter_clone.changed(gtk::FilterChange::LessStrict);
        if !sw_clone.is_active() {
            col_saldo_clone.set_visible(true);
        }
    });

    let custfilter_clone = custfilter.clone();
    let wb_clone = wb.clone();
    let todos_clone = todos.clone();
    sw.connect_active_notify(move |sw| {
        if sw.is_active() {
            wb_clone.set_sensitive(true);
            custfilter_clone.changed(gtk::FilterChange::MoreStrict);
            col_saldo.set_visible(false);
        } else {
            wb_clone.set_sensitive(false);
            custfilter_clone.changed(gtk::FilterChange::LessStrict);
            if todos_clone.is_active() {
                col_saldo.set_visible(true);
            }
        }
    });
    let filtros_clone = filtros.clone();
    pop.connect_map(move |_| {
        filtros_clone.remove_css_class("warning");
    });
    let filtros_clone = filtros.clone();
    pop.connect_closed(move |_| {
        if sw.is_active() || !todos.is_active() {
            filtros_clone.add_css_class("warning");
        }
    });
    let conn_clone = conn.clone();
    let custfilter_clone = custfilter.clone();
    let store_clone = store.clone();
    let wb_clone = wb.clone();
    gest.connect_clicked(move |_| {
        pop.popdown();
        etiquetas(&conn_clone, &wb_clone, &custfilter_clone , &store_clone, None);
    });
}

fn crea_menu(mca: &gtk::MenuButton, conn: &Rc<rusqlite::Connection>, nav: &adw::NavigationView, window: &adw::ApplicationWindow, store: &gtk::gio::ListStore, store_af: &gtk::gio::ListStore, dd: &gtk::DropDown, wb: &adw::WrapBox, custfilter: &gtk::CustomFilter, stack: &adw::ViewStack, searchentry: &gtk::SearchEntry, fstore: &gtk::FilterListModel, cv: &gtk::ColumnView) {
    let ops = ["Movimiento", "Transferencia", "Cuadrar caja"];
    let menu = gtk::gio::Menu::new();
    for op in ops {
        let action_name: String = op.chars().filter(|&c| !c.is_whitespace()).collect();
        let action = gtk::gio::SimpleAction::new(&format!("{}_ca", action_name), None);
        window.add_action(&action);
        menu.append(Some(&op), Some(&format!("win.{}_ca", action_name)));
        if op == "Movimiento" {
            let conn_clone = conn.clone();
            action.connect_activate(glib::clone!(
                #[weak]
                nav,
                #[weak]
                window,
                #[weak]
                store,
                #[weak]
                wb,
                #[weak]
                custfilter,
                #[weak]
                cv,
                move |_, _| {
                    pages_ca::mov(&conn_clone, &nav, &window, &store, &wb, &custfilter, None, Some(&cv));
                }
            ));
        } else if op == "Transferencia" {
            let conn_clone = conn.clone();
            action.connect_activate(glib::clone!(
                #[weak]
                nav,
                #[weak]
                window,
                #[weak]
                store,
                #[weak]
                cv,
                move |_, _| {
                    pages_ca::transf(&conn_clone, &nav, &window, &store, &cv);
                }
            ));
        } else if op == "Cuadrar caja" {
            let conn_clone = conn.clone();
            action.connect_activate(glib::clone!(
                #[weak]
                nav,
                #[weak]
                window,
                #[weak]
                store,
                #[weak]
                cv,
                move |_, _| {
                    pages_ca::cuadrar(&conn_clone, &nav, &window, &store, &cv);
                }
            ));
        }
    }
    let ops = ["Crear caja", "Renombrar caja", "Eliminar caja"];
    let sub = gtk::gio::Menu::new();
    for op in ops {
        let action_name: String = op.chars().filter(|&c| !c.is_whitespace()).collect();
        let action = gtk::gio::SimpleAction::new(&format!("{}_ca", action_name), None);
        window.add_action(&action);
        sub.append(Some(&op), Some(&format!("win.{}_ca", action_name)));
        if op == "Crear caja" {
            let conn_clone = conn.clone();
            action.connect_activate(glib::clone!(
                #[weak]
                nav,
                #[weak]
                window,
                #[weak]
                dd,
                move |_, _| {
                    pages_ca::crear_caja(&conn_clone, &nav, None, &window, None, Some(&dd), None, None);
                }
            ));
        } else if op == "Renombrar caja" {
            let conn_clone = conn.clone();
            action.connect_activate(glib::clone!(
                #[weak]
                window,
                #[weak]
                store,
                #[weak]
                dd,
                #[weak]
                nav,
                move |_, _| {
                    pages_ca::mod_caja(&conn_clone, &window, &store, &dd, &nav);
                }
            ));
        } else if op == "Eliminar caja" {
            let conn_clone = conn.clone();
            action.connect_activate(glib::clone!(
                #[weak]
                window,
                #[weak]
                store_af,
                #[weak]
                store,
                #[weak]
                dd,
                #[weak]
                stack,
                #[weak]
                nav,
                #[weak]
                searchentry,
                move |_, _| {
                    pages_ca::eliminar_caja(&conn_clone, &window, &store, &store_af, &dd, &stack, &nav, &searchentry);
                }
            ));
        }
    }
    menu.append_submenu(Some("Opciones caja"), &sub);
    let ops = ["Exportar csv", "Importar csv"];
    let sub = gtk::gio::Menu::new();
    for op in ops {
        let action_name: String = op.chars().filter(|&c| !c.is_whitespace()).collect();
        let action = gtk::gio::SimpleAction::new(&format!("{}_ca", action_name), None);
        window.add_action(&action);
        sub.append(Some(&op), Some(&format!("win.{}_ca", action_name)));
        if op == "Exportar csv" {
            action.connect_activate(glib::clone!(
                #[weak]
                window,
                #[weak]
                fstore,
                #[weak]
                cv,
                move |_, _| {
                    ie_csvs::exp_ca(&window, &fstore, &cv);
                }
            ));
        } else if op == "Importar csv" {
            let conn_clone = conn.clone();
            action.connect_activate(glib::clone!(
                #[weak]
                window,
                #[weak]
                wb,
                #[weak]
                store,
                #[weak]
                store_af,
                move |_, _| {
                    ie_csvs::imp_ca(&conn_clone, &window, &wb, &store, &store_af);
                }
            ));
        }
    }
    menu.append_submenu(Some("CSV"), &sub);
    let pop = gtk::PopoverMenu::from_model(Some(&menu));
    let cont_mca = adw::ButtonContent::builder()
        .icon_name("document-properties-symbolic")
        .label("Acciones")
        .build();
    mca.set_popover(Some(&pop));
    mca.set_child(Some(&cont_mca));
    mca.add_css_class("raised");
}

pub fn crea_lista(dd: &gtk::DropDown, store: &gtk::gio::ListStore, conn: &Rc<rusqlite::Connection>) {
    store.remove_all();
    let caja = dd.selected_item().unwrap().downcast::<gtk::StringObject>().unwrap().string();
    let mut stmt = conn.prepare(&format!("SELECT * FROM '{}' ORDER BY fecha, num", caja)).unwrap();
    let nc = stmt.column_count();
    let mut rows = stmt.query([]).unwrap();
    let mut saldo = 0.0;
    while let Some(row) = rows.next().unwrap() {
        let m = gtk::StringList::new(&[]);
        for i in 0..nc {
            let s;
            if i == 0 {
                let num : i32 = row.get(i).unwrap();
                s = num.to_string();
            } else if i == 1 {
                let fecha: String = row.get(i).unwrap();
                s = utils::convert_fech(&fecha);
            } else if i == 5 {
                let f: f32 = row.get(i).unwrap();
                saldo += f;
                s = format!("{:.2} €", f).replace(".", ",");
            } else {
                s = row.get(i).unwrap_or_else(|_| "".to_string());
            }
            m.append(&s);
        }
        let saldo_s = format!("{:.2} €", saldo).replace(".", ",");
        m.append(&saldo_s);
        store.insert(0, &m);
    }
    let m = gtk::StringList::new(&[&caja]);
    store.insert(0, &m);
}

fn dif_meses(desde: &str, hasta: &str) -> i64 {
    if utils::check_date(&desde) && desde != "" && utils::check_date(&hasta) && hasta != "" {
        let dayd: i32 = desde[0..2].parse().unwrap();
        let monthd: i32 = desde[3..5].parse().unwrap();
        let yeard: i32 = desde[6..10].parse().unwrap();
        let d = glib::DateTime::from_utc(yeard, monthd, dayd, 0, 0 , 0.0).unwrap();
        let dayh = hasta[0..2].parse().unwrap();
        let monthh = hasta[3..5].parse().unwrap();
        let yearh = hasta[6..10].parse().unwrap();
        let h = glib::DateTime::from_utc(yearh, monthh, dayh, 0, 0 , 0.0).unwrap();
        h.difference(&d).as_days()/30
    }
    else { 0 }
}

pub fn etiquetas(conn: &Rc<rusqlite::Connection>, wb: &adw::WrapBox, custfilter: &gtk::CustomFilter, store: &gtk::gio::ListStore, owbm: Option<&adw::WrapBox>) {
    let mov = if owbm.is_some() { true } else { false };
    let wbm = match owbm {
        Some(x) => x,
        None => &adw::WrapBox::new(),
    };
    let window = wb.root().and_downcast::<adw::ApplicationWindow>().unwrap();
    let pd = adw::PreferencesDialog::new();
    pd.set_title("Etiquetas");
    let pp = adw::PreferencesPage::new();
    let pg = adw::PreferencesGroup::new();
    let ent = adw::EntryRow::builder()
        .title("Nueva etiqueta")
        .show_apply_button(true)
        .build();
    pg.add(&ent);
    let chs = wb.observe_children();
    for i in 1..chs.n_items() {
        let tglab = chs.item(i).and_downcast::<gtk::ToggleButton>().unwrap().label().unwrap();
        let ar = crea_et_row(&tglab, &conn, &wb, &pd, &store, &wbm);
        pg.add(&ar);
    }
    pp.add(&pg);
    pd.add(&pp);
    pd.present(Some(&window));
    let conn_clone = conn.clone();
    let wb_clone = wb.clone();
    let custfilter_clone = custfilter.clone();
    let store_clone = store.clone();
    let wbm_clone = wbm.clone();
    ent.connect_apply(move |ent| {
        let toast = adw::Toast::new("");
        if ent.text().trim() == "" {
            toast.set_title("Debe tener un nombre.");
            ent.add_css_class("error");
            ent.grab_focus();
        } else if ent.text().contains('|') {
            toast.set_title("El nombre no puede contener '|'.");
            ent.add_css_class("error");
            ent.grab_focus();
        } else {
            let r = conn_clone.execute("INSERT INTO etiquetas (nombre) VALUES (?1)", [ent.text().to_string()]).unwrap_or_else(|_| { 0 });
            if r == 0 {
                toast.set_title("Ya existe la etiqueta.");
                ent.add_css_class("error");
                ent.grab_focus();
            } else {
                let tg = gtk::ToggleButton::with_label(&ent.text());
                wb_clone.append(&tg);
                if mov {
                    let tgm = gtk::ToggleButton::with_label(&ent.text());
                    wbm_clone.append(&tgm);
                }
                let custfilter_clone = custfilter_clone.clone();
                tg.connect_toggled(move |_| {
                    custfilter_clone.changed(gtk::FilterChange::Different);
                });
                let ar = crea_et_row(&ent.text(), &conn_clone, &wb_clone, &pd, &store_clone, &wbm_clone);
                pg.add(&ar);
                ar.grab_focus();
                ent.remove_css_class("error");
                ent.set_text("");
                return;
            }
        }
        toast.set_timeout(2);
        pd.add_toast(toast);
    });
}

fn crea_et_row(tit: &str, conn: &Rc<rusqlite::Connection>, wb: &adw::WrapBox, pd: &adw::PreferencesDialog, store: &gtk::gio::ListStore, wbm: &adw::WrapBox) -> adw::ActionRow {
    let hbox = gtk::Box::new(gtk::Orientation::Horizontal, 0);
    let edit = gtk::Button::builder()
        .icon_name("document-edit-symbolic")
        .valign(gtk::Align::Center)
        .css_classes(["flat"])
        .tooltip_text("Renombrar")
        .build();
    hbox.append(&edit);
    let del = gtk::Button::builder()
        .icon_name("edit-delete-symbolic")
        .valign(gtk::Align::Center)
        .css_classes(["flat"])
        .tooltip_text("Eliminar")
        .build();
    hbox.append(&del);
    let ar = adw::ActionRow::builder()
        .title(tit)
        .build();
    ar.add_suffix(&hbox);
    let pd_clone = pd.clone();
    let ar_clone = ar.clone();
    let wb_clone = wb.clone();
    let conn_clone = conn.clone();
    let store_clone = store.clone();
    let wbm_clone = wbm.clone();
    edit.connect_clicked(move |_| {
        let ad = adw::AlertDialog::new(Some("Renombrar etiqueta"), Some(&ar_clone.title()));
        ad.add_responses(&[("canc", "Cancelar"), ("acept", "Aceptar")]);
        ad.set_default_response(Some("acept"));
        ad.set_response_appearance("acept", adw::ResponseAppearance::Suggested);
        let pgs = adw::PreferencesGroup::new();
        let nombre = adw::EntryRow::builder()
            .title("Nuevo nombre")
            .activates_default(true)
            .build();
        pgs.add(&nombre);
        let aviso = gtk::Label::new(Some("Se renombrará la etiqueta en todos los movimientos."));
        pgs.add(&aviso);
        ad.set_extra_child(Some(&pgs));
        ad.set_response_enabled("acept", false);
        ad.present(Some(&pd_clone));
        let ar_clone = ar_clone.clone();
        let nombre_clone = nombre.clone();
        let wb_cloner = wb_clone.clone();
        let conn_clone = conn_clone.clone();
        let store_clone = store_clone.clone();
        let wbm_clone = wbm_clone.clone();
        ad.connect_response(Some("acept"), move |_, _| {
            let nombre_ant = ar_clone.title();
            ar_clone.set_title(&nombre_clone.text());
            let i = ar_clone.index();
            let chs = wb_cloner.observe_children();
            chs.item(i as u32).and_downcast::<gtk::ToggleButton>().unwrap().set_label(&nombre_clone.text());
            let chs = wbm_clone.observe_children();
            if chs.n_items() != 0 {
                chs.item(i as u32 -1).and_downcast::<gtk::ToggleButton>().unwrap().set_label(&nombre_clone.text());
            }
            conn_clone.execute("UPDATE etiquetas SET nombre=?1 WHERE nombre=?2", [nombre_clone.text().to_string(), nombre_ant.to_string()]).unwrap();
            comun_edit_elim_et(&store_clone, &nombre_ant, &nombre_clone.text(), &conn_clone);
        });
        let wb_clone = wb_clone.clone();
        nombre.connect_changed(move |nombre| {
            if nombre.text().trim() == "" {
                ad.set_response_enabled("acept", false);
            } else if nombre.text().contains('|') {
                ad.set_response_enabled("acept", false);
            } else {
                let mut r = true;
                let chs = wb_clone.observe_children();
                for i in 0..chs.n_items() {
                    let tglab = chs.item(i).and_downcast::<gtk::ToggleButton>().unwrap().label().unwrap();
                    if tglab == nombre.text() {
                        r = false;
                        break;
                    }
                }                
                ad.set_response_enabled("acept", r);
            }
        });
    });
    let pd_clone = pd.clone();
    let ar_clone = ar.clone();
    let wb_clone = wb.clone();
    let conn_clone = conn.clone();
    let store_clone = store.clone();
    let wbm_clone = wbm.clone();
    del.connect_clicked(move |_| {
        let ad = adw::AlertDialog::new(Some("Eliminar etiqueta"), Some("Se eliminará de todos los movimientos."));
        ad.add_responses(&[("canc", "Cancelar"), ("acept", "Aceptar")]);
        ad.set_default_response(Some("acept"));
        ad.set_response_appearance("acept", adw::ResponseAppearance::Destructive);
        let pgs = adw::PreferencesGroup::new();
        ad.set_extra_child(Some(&pgs));
        ad.present(Some(&pd_clone));
        let ar_clone = ar_clone.clone();
        let wb_clone = wb_clone.clone();
        let conn_clone = conn_clone.clone();
        let store_clone = store_clone.clone();
        let wbm_clone = wbm_clone.clone();
        ad.connect_response(Some("acept"), move |_, _| {
            let nombre = ar_clone.title();
            let i = ar_clone.index();
            let lb = ar_clone.parent().and_downcast::<gtk::ListBox>().unwrap();
            lb.remove(&ar_clone);
            let chs = wb_clone.observe_children();
            let tg = chs.item(i as u32).and_downcast::<gtk::ToggleButton>().unwrap();
            wb_clone.remove(&tg);
            let chs = wbm_clone.observe_children();
            if chs.n_items() != 0 {
                let tg = chs.item(i as u32 -1).and_downcast::<gtk::ToggleButton>().unwrap();
                wbm_clone.remove(&tg);
            }
            conn_clone.execute("DELETE FROM etiquetas WHERE nombre = ?1", [nombre.to_string()]).unwrap();
            comun_edit_elim_et(&store_clone, &nombre, "", &conn_clone);
        });
    });
    ar
}

fn comun_edit_elim_et(store: &gtk::gio::ListStore, nombre: &str, nuevo: &str, conn: &Rc<rusqlite::Connection>) {
    for i in 1..store.n_items() {
        let sli = store.item(i).and_downcast::<gtk::StringList>().unwrap();
        let ets = sli.string(4).unwrap();
        let etsf = mod_etiq_str(&ets, nombre, nuevo);
        sli.splice(4, 1, &[&etsf]);
        //Utilizo remove + insert porque con splice no se actualiza automáticamente la lista
        store.remove(i);
        store.insert(i, &sli);
    }
    let sl_cajas = db_utils::lista_cajas(&conn);
    let tx = conn.unchecked_transaction().unwrap();
    for i in 0..sl_cajas.n_items() {
        let query = format!("SELECT num, etiquetas FROM '{}' WHERE etiquetas LIKE '%{}%'", sl_cajas.string(i).unwrap(), nombre);
        let mut stmt = tx.prepare(&query).unwrap();
        let mut rows = stmt.query([]).unwrap();
        while let Some(row) = rows.next().unwrap() {
            let num: u32 = row.get(0).unwrap();
            let ets: String = row.get(1).unwrap();
            let etsf = mod_etiq_str(&ets, nombre, nuevo);
            let upd = format!("UPDATE '{}' SET etiquetas='{}' WHERE num={}", sl_cajas.string(i).unwrap(), etsf, num);
            tx.execute(&upd, []).unwrap();
        }
    }
    let _ = tx.commit();    
}

fn mod_etiq_str(ets: &str, nombre: &str, nuevo: &str) -> String {
    let mut iter = ets.split(" | ");
    let mut etsf = String::new();
    let mut first = true;
    while let Some(et) = iter.next() {
        if et != nombre && !first {
            etsf.push_str(" | ");
            etsf.push_str(et);
        } else if et != nombre && first {
            etsf.push_str(et);
            first = false;
        } else if nuevo != "" && !first {
            etsf.push_str(" | ");
            etsf.push_str(nuevo);
        } else if nuevo != "" {
            etsf.push_str(nuevo);
            first = false;
        }
    }
    etsf
}
